package ru.yandex.crypta.lab.tables;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.Configuration;
import org.jooq.Delete;
import org.jooq.Field;
import org.jooq.Insert;
import org.jooq.InsertValuesStep3;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.Select;
import org.jooq.SelectHavingStep;
import org.jooq.Table;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.Subsamples;


public class SubsamplesTable extends GenericTable<Subsamples> {
    public static final Table<Record> TABLE = DSL.table("api_sample_user_set_ids");

    public static final Field<String> SAMPLE_ID =
            DSL.field(DSL.name(TABLE.getName(), "sample_id"), String.class);
    public static final Field<String> USER_SET_ID =
            DSL.field(DSL.name(TABLE.getName(), "user_set_id"), String.class);
    public static final Field<String> GROUPING_KEY_VALUE
            = DSL.field(DSL.name(TABLE.getName(), "grouping_key_value"), String.class);

    public static final Field<String[][]> SUBSAMPLE_IDS = DSL.arrayAgg(DSL.array(USER_SET_ID, GROUPING_KEY_VALUE));

    public SubsamplesTable(Configuration configuration) {
        super(configuration, Subsamples.class);
    }

    public static Map<String, String> getSubsampleIdsBuilders(Record record) {
        return Arrays.stream(record.get(SUBSAMPLE_IDS))
                .filter(ids -> (ids[0] != null) && (ids[1] != null))
                .collect(Collectors.toMap(
                        ids -> ids[0],
                        ids -> ids[1]
                ));
    }

    @Override
    public Subsamples read(Record record) {
        return Subsamples.newBuilder()
                .setSampleId(record.get(SAMPLE_ID))
                .putAllUserSetIdToGroupingKeyValue(getSubsampleIdsBuilders(record))
                .build();
    }

    @Override
    public Select<Record> selectQuery() {
        return dsl.selectFrom(TABLE);
    }

    public SelectHavingStep<Record2<String, String[][]>> selectBySampleIdQuery(String sampleId) {
        return dsl.select(SAMPLE_ID, SUBSAMPLE_IDS)
                .from(TABLE)
                .where(SAMPLE_ID.eq(sampleId))
                .groupBy(SAMPLE_ID);
    }

    public Insert<Record> insertQuery(Subsamples subsample) {
        InsertValuesStep3<Record, String, String, String> insertValueStep =  dsl.insertInto(TABLE, SAMPLE_ID, USER_SET_ID, GROUPING_KEY_VALUE);

        subsample.getUserSetIdToGroupingKeyValueMap()
                .forEach((key, value) -> insertValueStep.values(subsample.getSampleId(), key, value));

        return insertValueStep;
    }

    public Delete<Record> deleteBySampleIdQuery(String sampleId) {
        return dsl.deleteFrom(TABLE).where(SAMPLE_ID.eq(sampleId));
    }

}
