package ru.yandex.crypta.lab.tables;

import org.jooq.Configuration;
import org.jooq.Delete;
import org.jooq.Field;
import org.jooq.Insert;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.impl.DSL;

import ru.yandex.crypta.common.data.GenericTable;
import ru.yandex.crypta.lab.proto.ETrainableSegmentOriginType;
import ru.yandex.crypta.lab.proto.ETrainableSegmentTargetType;
import ru.yandex.crypta.lab.proto.TTrainableSegment;


public class TrainableSegmentsTable extends GenericTable<TTrainableSegment> {
    public static final Table<Record> TABLE = DSL.table("lab_training_sample_user_set_ids");

    public static final Field<String> SAMPLE_ID = DSL.field(DSL.name(TABLE.getName(), "sample_id"), String.class);
    public static final Field<String> USER_SET_ID = DSL.field(DSL.name(TABLE.getName(), "user_set_id"), String.class);
    public static final Field<String> TARGET_TYPE = DSL.field(DSL.name(TABLE.getName(), "target_type"), String.class);
    public static final Field<String> ORIGIN_TYPE = DSL.field(DSL.name(TABLE.getName(), "origin_type"), String.class);
    public static final Field<Integer> SEGMENT_SIZE = DSL.field(DSL.name(TABLE.getName(), "segment_size"), Integer.class);

    public static final Long DEFAULT_POSITIVE_SEGMENT_SIZE = 1_000_000L;
    public static final Long DEFAULT_NEGATIVE_SEGMENT_SIZE = 1_000_000L;

    public TrainableSegmentsTable(Configuration configuration) {
        super(configuration, TTrainableSegment.class);
    }

    @Override
    protected TTrainableSegment read(Record record) {
        return TTrainableSegment.newBuilder()
                .setSampleId(record.get(SAMPLE_ID))
                .setSiberiaUserSetId(record.get(USER_SET_ID))
                .setName((record.get(ORIGIN_TYPE) + "_" + record.get(TARGET_TYPE)).toLowerCase())
                .setTargetType(ETrainableSegmentTargetType.valueOf(record.get(TARGET_TYPE)))
                .setOriginType(ETrainableSegmentOriginType.valueOf(record.get(ORIGIN_TYPE)))
                .setSegmentSize(record.get(SEGMENT_SIZE))
                .build();
    }

    @Override
    public Select<Record>  selectQuery() { return dsl.selectFrom(TABLE); }

    public Insert<Record> insertQuery(TTrainableSegment segment) {
        return dsl.insertInto(TABLE)
                .set(SAMPLE_ID, segment.getSampleId())
                .set(USER_SET_ID, segment.getSiberiaUserSetId())
                .set(TARGET_TYPE, segment.getTargetType().name())
                .set(ORIGIN_TYPE, segment.getOriginType().name())
                .set(SEGMENT_SIZE, segment.getSegmentSize());
    }

    public Delete<Record> deleteBySampleIdQuery(String sampleId) {
        return dsl.deleteFrom(TABLE).where(SAMPLE_ID.eq(sampleId));
    }

    public Select<Record> selectBySampleIdQuery(String sampleId) {
        return dsl.selectFrom(TABLE).where(SAMPLE_ID.eq(sampleId));
    }
}
