package ru.yandex.crypta.lab.utils;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.ws.rs.core.SecurityContext;

import ru.yandex.crypta.audience.proto.TUserDataStats;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.proto.AccessLevel;
import ru.yandex.crypta.lab.proto.EHashingMethod;
import ru.yandex.crypta.lab.proto.ELabIdentifierType;
import ru.yandex.crypta.lab.proto.Sample;
import ru.yandex.crypta.lab.proto.TMatchingOptions;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.common.YtErrorMapping.AuthorizationError;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

public class Acl {

    public static final int MIN_SAMPLE_SIZE = 100;

    public static final Set<Roles.Lab.Matching.Mode> ONLY_MATCHING =
            Set.of(Roles.Lab.Matching.Mode.MATCHING_SIDE_BY_SIDE);
    public static final Set<Roles.Lab.Matching.Mode> MATCHING_OR_CONVERTING = Set.of(
            Roles.Lab.Matching.Mode.MATCHING_SIDE_BY_SIDE,
            Roles.Lab.Matching.Mode.CONVERTING_GROUP
    );

    public static final Set<Roles.Lab.Matching.Hashing> ONLY_NON_HASHED = Set.of(Roles.Lab.Matching.Hashing.NON_HASHED);
    public static final Set<Roles.Lab.Matching.Hashing> HASHED_OR_NON_HASHED = Set.of(
            Roles.Lab.Matching.Hashing.HASHED,
            Roles.Lab.Matching.Hashing.NON_HASHED
    );

    private static final Set<ELabIdentifierType> PRIVATE_IDENTIFIERS = EnumSet.of(
            ELabIdentifierType.LAB_ID_PHONE,
            ELabIdentifierType.LAB_ID_EMAIL,
            ELabIdentifierType.LAB_ID_LOGIN,
            ELabIdentifierType.LAB_ID_PUID
    );

    private static final String EFFECTIVE_ACL = "effective_acl";
    private static final String ACTION = "action";
    private static final String PERMISSIONS = "permissions";
    private static final String SUBJECTS = "subjects";
    private static final String READ = "read";
    private static final String YANDEX = "yandex";


    private Acl() {

    }

    public static List<YTreeNode> get(Cypress cypress, Optional<GUID> transaction, YPath path) {
        if (!cypress.exists(transaction, YtParameters.DO_NOT_PING, path)) {
            throw Exceptions.wrongRequestException("No such table: " + path.toString(), "TABLE_IS_MISSED");
        }
        try {
            YTreeNode ytObject = cypress.get(transaction, false, path, Set.of(Acl.EFFECTIVE_ACL));
            return ytObject.getAttribute(Acl.EFFECTIVE_ACL).get().asList();
        } catch (AuthorizationError e) {
            throw Exceptions
                    .wrongRequestException("Read permission for node: " + path.toString() + "is not allowed by ACL",
                            "ACCESS_DENIED");
        }
    }

    public static YTreeListNode readOnly(List<YTreeNode> acl) {
        YTreeBuilder builder = new YTreeBuilder().beginList();
        acl.forEach(each -> {
            Map<String, YTreeNode> aclElement = each.asMap();
            boolean hasRead = aclElement.get(Acl.PERMISSIONS)
                    .asList()
                    .stream()
                    .anyMatch(x -> x.stringValue().equals(Acl.READ));
            String action = aclElement.get(Acl.ACTION).stringValue();
            YTreeListNode subjects = aclElement.get(Acl.SUBJECTS).listNode();
            if (hasRead) {
                builder.beginMap()
                        .key(Acl.ACTION).value(action)
                        .key(Acl.PERMISSIONS).value(YTree.listBuilder().value(Acl.READ).buildList())
                        .key(Acl.SUBJECTS).value(subjects)
                        .endMap();
            }
        });
        return builder.buildList();
    }

    public static YTreeListNode publicAcl() {
        YTreeBuilder builder = new YTreeBuilder().beginList();
        builder.beginMap()
                .key(Acl.ACTION).value("allow")
                .key(Acl.PERMISSIONS).value(YTree.listBuilder().value(Acl.READ).buildList())
                .key(Acl.SUBJECTS).value(YTree.listBuilder().value(Acl.YANDEX).buildList())
                .endMap();
        return builder.buildList();
    }

    public static YTreeListNode loginAcl(String login) {
        YTreeBuilder builder = new YTreeBuilder().beginList();
        builder.beginMap()
                .key(Acl.ACTION).value("allow")
                .key(Acl.PERMISSIONS).value(YTree.listBuilder().value(Acl.READ).buildList())
                .key(Acl.SUBJECTS).value(YTree.listBuilder().value(Acl.YANDEX).value(login).buildList())
                .endMap();
        return builder.buildList();
    }

    public static boolean canCreateViewsForSample(Sample sample, SecurityContext securityContext) {
        boolean isAuthor = Objects.equals(sample.getAuthor(), securityContext.getUserPrincipal().getName());
        return isAdmin(securityContext) || isAuthor;
    }

    private static boolean isAdmin(SecurityContext securityContext) {
        return securityContext.isUserInRole(Roles.ADMIN) || securityContext.isUserInRole(Roles.Lab.ADMIN);
    }

    public static boolean canViewStats(TUserDataStats stats, SecurityContext securityContext) {
        return (stats.getCounts().getUniqYuid() >= MIN_SAMPLE_SIZE) || isAdmin(securityContext);
    }

    private static Function<Roles.Lab.Matching.Type, Set<String>> combo(
            Set<Roles.Lab.Matching.Hashing> hashings,
            Set<Roles.Lab.Matching.Mode> modes
    ) {
        return (Roles.Lab.Matching.Type idType) -> {
            HashSet<String> result = new HashSet<>(hashings.size() * modes.size());
            for (Roles.Lab.Matching.Hashing eachHashing : hashings) {
                for (Roles.Lab.Matching.Mode eachMode : modes) {
                    Roles.Lab.Matching.Privacy privacyType = Roles.Lab.Matching.Privacy.PRIVATE;
                    String role = Roles.Lab.Matching.role(privacyType, eachMode, eachHashing, idType);
                    result.add(role);
                }
            }
            return result;
        };
    }

    public static Set<String> sufficientRoles(TMatchingOptions options) {
        boolean withHashing = !options.getHashingMethod().equals(EHashingMethod.HM_IDENTITY);
        boolean isPrivateIdentifier = PRIVATE_IDENTIFIERS.contains(options.getIdType());
        boolean isMatching = options.getIncludeOriginal();

        if (!isPrivateIdentifier) {
            return Collections.singleton(Roles.Lab.Matching.NON_PRIVATE_MATCHING);
        } else {
            Set<Roles.Lab.Matching.Hashing> needAnyOfHashings =
                    withHashing ? HASHED_OR_NON_HASHED : ONLY_NON_HASHED;
            Set<Roles.Lab.Matching.Mode> needAnyOfModes =
                    isMatching ? ONLY_MATCHING : MATCHING_OR_CONVERTING;

            Function<Roles.Lab.Matching.Type, Set<String>> sufficientAnyRoleForType =
                    combo(needAnyOfHashings, needAnyOfModes);

            switch (options.getIdType()) {
                case LAB_ID_EMAIL:
                    return sufficientAnyRoleForType.apply(Roles.Lab.Matching.Type.EMAIL);
                case LAB_ID_PHONE:
                    return sufficientAnyRoleForType.apply(Roles.Lab.Matching.Type.PHONE);
                case LAB_ID_LOGIN:
                    return sufficientAnyRoleForType.apply(Roles.Lab.Matching.Type.LOGIN);
                case LAB_ID_PUID:
                    return sufficientAnyRoleForType.apply(Roles.Lab.Matching.Type.PUID);
                default:
                    return Collections.emptySet();
            }
        }
    }

    public static long getMinimalSize(TMatchingOptions options) {
        boolean withHashing = !options.getHashingMethod().equals(EHashingMethod.HM_IDENTITY);
        boolean isPrivateIdentifier = PRIVATE_IDENTIFIERS.contains(options.getIdType());
        boolean isMatching = options.getIncludeOriginal();

        if (!isPrivateIdentifier) {
            if (isMatching) {
                return 0;
            } else {
                return 0;
            }
        } else {
            if (withHashing) {
                return 0;
            } else {
                return 1000;
            }
        }
    }

    public static boolean canCreateSuchView(TMatchingOptions options, SecurityContext securityContext) {
        if (isAdmin(securityContext)) {
            return true;
        }
        return sufficientRoles(options).stream().anyMatch(securityContext::isUserInRole);
    }

    public static YTreeListNode getDirectoryAcl(List<YTreeNode> originalAcl, AccessLevel accessLevel) {
        if (Objects.equals(accessLevel, AccessLevel.PUBLIC)) {
            return publicAcl();
        } else {
            return readOnly(originalAcl);
        }
    }
}
