package ru.yandex.crypta.lab.utils;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.google.common.net.InternetDomainName;

import ru.yandex.crypta.lab.proto.Constructor;
import ru.yandex.crypta.lab.proto.FullValue;
import ru.yandex.crypta.lab.proto.RuleCondition;

public class RuleApproval {
    static private final List<String> DOMAINS_WITH_APPROVAL_REQUIRED = List.of("yandex", "auto.ru", "kinopoisk.ru", "k50.ru", "adfox.ru", "edadeal", "autoru");
    static private final String APPROVAL_REQUIRED_SITE_REGEX = String.format(".*(%s).*", String.join("|", DOMAINS_WITH_APPROVAL_REQUIRED).replace(".", "\\."));

    static private final Set<RuleCondition.Source> PUBLIC_SITE_RULES = Set.of(
            RuleCondition.Source.PUBLIC_SITES,
            RuleCondition.Source.YANDEX_REFERRER,
            RuleCondition.Source.BROWSER_SITES
    );

    static private boolean urlsNeedApproval(List<String> urls) {
        var domains = new HashSet<String>();
        try {
            for (var url: urls) {
                if (url.startsWith("regexp:")) {
                    return true;
                }
                if (!url.contains("://")) {
                    url = "https://" + url;
                }
                var parsed = new URL(url);
                if (parsed.getHost().toLowerCase().matches(APPROVAL_REQUIRED_SITE_REGEX)) {
                    return true;
                }

                var fullDomain = InternetDomainName.from(parsed.getHost());
                domains.add(
                    fullDomain.isUnderPublicSuffix() ?
                    fullDomain.topPrivateDomain().parts().get(0) :
                    parsed.getHost()
                );
            }
        } catch (MalformedURLException e) {
            return false;
        }
        return domains.size() == 1;
    }

    static private boolean publicSitesNeedApproval(List<FullValue> fullValues) {
        return urlsNeedApproval(fullValues.stream().map(FullValue::getNormalized).collect(Collectors.toList()));
    }

    static private boolean appsNeedApproval(List<FullValue> fullValues) {
        return urlsNeedApproval(fullValues.stream().map(
                x -> String.join(".", Lists.reverse(Arrays.asList(x.getNormalized().split("\\."))))
        ).collect(Collectors.toList()));
    }

    static public RuleCondition.State getState(RuleCondition.Source source, List<FullValue> fullValues) {
        if (source.getValueDescriptor().getOptions().getExtension(Constructor.needsApprove)) {
            return RuleCondition.State.NEED_APPROVE;
        } else if (PUBLIC_SITE_RULES.contains(source)) {
            return publicSitesNeedApproval(fullValues) ?
                    RuleCondition.State.NEED_APPROVE :
                    RuleCondition.State.APPROVED;
        } else if (source == RuleCondition.Source.APPS) {
            return appsNeedApproval(fullValues) ? RuleCondition.State.NEED_APPROVE : RuleCondition.State.APPROVED;
        } else {
            return RuleCondition.State.APPROVED;
        }
    }
}
