package ru.yandex.crypta.lab.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.lib.proto.EEnvironment;

public class Tanker {

    private static final String TANKER_API_GET_KEYS_URL =
            "https://tanker-api.yandex-team.ru/keysets/tjson/?keyset-id=profile&project-id=crypta";
    private static final String TANKER_API_ADD_KEY_URL = "https://tanker-api.yandex-team.ru/keysets/merge/";
    private static final String TANKER_API_DELETE_KEY_URL = "https://tanker-api.yandex-team.ru/v2/keys/delete/";
    private static final Logger LOG = LoggerFactory.getLogger(Tanker.class);
    private final String token;
    private final EEnvironment environment;

    public Tanker(EEnvironment environment, String token) {
        this.environment = environment;
        this.token = token;
    }

    private boolean isTestingEnvironment() {
        return Objects.equals(environment, EEnvironment.ENV_TESTING);
    }

    public void sendPostRequestWithFile(String filePath) {
        try {
            CloseableHttpClient httpClient = HttpClients.createDefault();
            HttpPost httpPost = new HttpPost(TANKER_API_ADD_KEY_URL);
            httpPost.addHeader("AUTHORIZATION", String.format("OAuth %s", token));
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.addTextBody("language", "ru", ContentType.TEXT_PLAIN);
            builder.addTextBody("format", "tjson", ContentType.TEXT_PLAIN);
            builder.addTextBody("project-id", "crypta", ContentType.TEXT_PLAIN);
            builder.addTextBody("keyset-id", "profile", ContentType.TEXT_PLAIN);
            if (isTestingEnvironment()) {
                builder.addTextBody("branch-id", "testing", ContentType.TEXT_PLAIN);
            }

            LOG.debug("sendPostRequestWithFile: using {}", filePath);

            // This attaches the file to the POST:
            File f = new File(filePath);
            builder.addBinaryBody(
                    "file",
                    new FileInputStream(f),
                    ContentType.APPLICATION_JSON,
                    f.getName()
            );

            HttpEntity multipart = builder.build();
            httpPost.setEntity(multipart);
            CloseableHttpResponse response = httpClient.execute(httpPost);
            LOG.info("sendPostRequestWithFile: Tanker API response is {}", response.getStatusLine().toString());
        } catch (Exception e) {
            LOG.error("sendPostRequestWithFile: {}", e.toString());
        }
    }

    public String createTankerCreateSegmentJson(String name, String value, boolean skipTranslation) {
        LOG.debug("createTankerCreateSegmentJson: name={} value={}", name, value);

        final JsonNodeFactory factory = JsonNodeFactory.instance;

        final ObjectNode tankerKeyCreateMessage = factory.objectNode();
        final ObjectNode tankerKeysets = factory.objectNode();
        final ObjectNode tankerProfile = factory.objectNode();
        final ObjectNode tankerKeys = factory.objectNode();
        final ObjectNode tankerKey = factory.objectNode();
        final ObjectNode tankerTranslations = factory.objectNode();
        final ObjectNode tankerRuTranslation = factory.objectNode();
        final ObjectNode tankerMeta = factory.objectNode();
        final ArrayNode tankerLanguages = factory.arrayNode();

        if (skipTranslation) {
            tankerRuTranslation.put("status", "approved");
        } else {
            tankerRuTranslation.put("status", "translated");
        }

        tankerRuTranslation.put("translator_comment", "");
        tankerRuTranslation.put("change_date", "");
        tankerRuTranslation.put("form", value);
        tankerTranslations.set("ru", tankerRuTranslation);
        tankerKey.set("translations", tankerTranslations);
        tankerKeys.set(name, tankerKey);
        tankerProfile.set("keys", tankerKeys);
        tankerKeysets.set("profile", tankerProfile);
        tankerKeyCreateMessage.set("keysets", tankerKeysets);
        tankerLanguages.add("ru");
        tankerMeta.set("languages", tankerLanguages);
        tankerKeyCreateMessage.set("meta", tankerMeta);
        return tankerKeyCreateMessage.toString();
    }

    public String getTankerResponse(String url) {
        try {
            StringBuilder result = new StringBuilder();

            URL endpointUrl = new URL(url);
            if (isTestingEnvironment()) {
                endpointUrl = new URL(String.format("%s&branch-id=testing", url));
            }

            HttpURLConnection conn = (HttpURLConnection) endpointUrl.openConnection();
            conn.setRequestProperty("Accept-Charset", "UTF-8");
            conn.setRequestMethod("GET");

            if (conn.getResponseCode() != 200) {
                throw Exceptions.unavailable();
            }

            BufferedReader rd = new BufferedReader(
                    new InputStreamReader(conn.getInputStream(), StandardCharsets.UTF_8)
            );
            String line;
            while ((line = rd.readLine()) != null) {
                result.append(line);
            }
            rd.close();

            return result.toString();
        } catch (IOException exception) {
            LOG.error("Failed to get response from Tanker");
            throw Exceptions.unavailable();
        }
    }

    public String getKeysFromTanker() {
        return getTankerResponse(TANKER_API_GET_KEYS_URL);
    }

    public String getTranslationsByKey(String tankerKey) {
        String url = String.format("%s&key-id=%s", TANKER_API_GET_KEYS_URL, tankerKey);

        return getTankerResponse(url);
    }

    public void deleteKeyFromTanker(String tankerKey) {
        OkHttpClient client = new OkHttpClient();

        String branch = "master";
        if (isTestingEnvironment()) {
            branch = "testing";
        }

        Map<String, String> payload = new HashMap<>();
        payload.put("keyset", "profile");
        payload.put("project", "crypta");
        payload.put("key", tankerKey);
        payload.put("branch", branch);

        try {
            RequestBody requestBody = RequestBody.create(
                    new ObjectMapper().writeValueAsString(payload), MediaType.parse(JsonUtf8.MEDIA_TYPE)
            );

            Request request = new Request.Builder()
                    .url(TANKER_API_DELETE_KEY_URL)
                    .header("Content-Type", "application/json")
                    .header("Authorization", String.format("OAuth %s", token))
                    .post(requestBody)
                    .build();

            // TODO: check response?
            client.newCall(request).execute().close();
        } catch (JsonProcessingException e) {
            throw Exceptions.unchecked(e);
        } catch (IOException e) {
            throw Exceptions.unavailable();
        }
    }


    public EEnvironment getEnvironment() {
        return environment;
    }

    public String getToken() {
        return token;
    }
}
