package ru.yandex.crypta.lab.yt;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;

import javax.inject.Inject;

import org.quartz.JobBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.common.ws.EntityId;
import ru.yandex.crypta.lab.AudienceService;
import ru.yandex.crypta.lab.base.BaseService;
import ru.yandex.crypta.lab.job.AudienceCreateJob;
import ru.yandex.crypta.lab.job.AudienceDeleteJob;
import ru.yandex.crypta.lab.job.AudienceModifyJob;
import ru.yandex.crypta.lab.proto.Audience;
import ru.yandex.crypta.lab.proto.Timestamps;
import ru.yandex.crypta.lab.tables.Tables;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.schedulers.QuartzScheduler;
import ru.yandex.crypta.lib.schedulers.Schedulers;


public class DefaultAudienceService extends BaseService<AudienceService> implements AudienceService {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultAudienceService.class);
    private final Schedulers schedulers;

    @Inject
    public DefaultAudienceService(EEnvironment environment, PostgresClient sql, Schedulers schedulers) {
        super(environment, sql);
        this.schedulers = schedulers;
    }

    private Audience fetchAccessibleLabAudience(String id, Tables tables) {
        return tables
                .audiences()
                .selectByIdAccessibleQuery(id)
                .fetchOptionalInto(Audience.class)
                .orElseThrow(NotFoundException::new);
    }

    private Audience fetchModifiableLabAudience(String id, Tables tables) {
        return tables
                .audiences()
                .selectByIdModifiableQuery(id)
                .fetchOptionalInto(Audience.class)
                .orElseThrow(NotFoundException::new);
    }

    @Override
    public Audience getLabAudienceById(String id) throws NotFoundException {
        return fetchAccessibleLabAudience(id, tables());
    }

    @Override
    public List<Audience> getMyLabAudiences() {
        return tables()
                .audiences()
                .selectMineQuery()
                .fetchInto(Audience.class);
    }

    private <T> T withAudienceForUpdate(String id, BiFunction<Tables, Audience.Builder, T> updater) {
        return withSqlTransaction(tables -> {
            Audience.Builder audience = fetchModifiableLabAudience(id, tables).toBuilder();
            return updater.apply(tables, audience);
        });
    }

    @Override
    public Audience createAudience(Audience.Builder prototype) {
        return withSqlTransaction(tables -> {
            String id = new EntityId("audience").toString();

            long timestamp = Instant.now().getEpochSecond();
            Audience audience = prototype
                    .setAuthor(securityContext().getUserPrincipal().getName())
                    .setId(id)
                    .setTimestamps(Timestamps.newBuilder()
                            .setCreated(timestamp)
                            .setModified(timestamp))
                    .setState(Audience.State.CREATING)
                    .build();

            tables.audiences().insertQuery(audience).execute();
            scheduleCreateAudienceSegment(audience);
            return audience;
        });
    }

    @Override
    public Audience setExternalId(String id, Long externalId) throws NotFoundException {
        return withAudienceForUpdate(id, (tables, segment) -> {
            Audience oldAudience = fetchModifiableLabAudience(id, tables);

            long timestamp = Instant.now().getEpochSecond();
            Audience newAudience = oldAudience.toBuilder()
                    .setTimestamps(Timestamps.newBuilder()
                            .setCreated(oldAudience.getTimestamps().getCreated())
                            .setModified(timestamp))
                    .setExternalId(externalId)
                    .build();
            tables.audiences().updateQuery(newAudience).execute();
            return newAudience;
        });
    }

    @Override
    public Audience updateState(String id, Audience.State state) throws NotFoundException {
        return withAudienceForUpdate(id, (tables, segment) -> {
            Audience oldAudience = fetchModifiableLabAudience(id, tables);

            long timestamp = Instant.now().getEpochSecond();
            Audience newAudience = oldAudience.toBuilder()
                    .setTimestamps(Timestamps.newBuilder()
                            .setCreated(oldAudience.getTimestamps().getCreated())
                            .setModified(timestamp))
                    .setState(state)
                    .build();
            tables.audiences().updateQuery(newAudience).execute();
            return newAudience;
        });
    }

    @Override
    public Audience modifyAudience(String id, String sourcePath, String sourceField) throws NotFoundException {
        return withAudienceForUpdate(id, (tables, segment) -> {
            Audience oldAudience = fetchModifiableLabAudience(id, tables);

            long timestamp = Instant.now().getEpochSecond();
            Audience newAudience = oldAudience.toBuilder()
                    .setTimestamps(Timestamps.newBuilder()
                            .setCreated(oldAudience.getTimestamps().getCreated())
                            .setModified(timestamp))
                    .setSourcePath(Optional.ofNullable(sourcePath).orElse(oldAudience.getSourcePath()))
                    .setSourceField(Optional.ofNullable(sourceField).orElse(oldAudience.getSourceField()))
                    .setState(Audience.State.MODIFYING)
                    .build();
            tables.audiences().updateQuery(newAudience).execute();

            scheduleModifyAudienceSegment(newAudience);
            return newAudience;
        });
    }


    @Override
    public Audience deleteAudience(String id) throws NotFoundException {
        return withAudienceForUpdate(id, (tables, segment) -> {
            Audience oldAudience = fetchModifiableLabAudience(id, tables);

            long timestamp = Instant.now().getEpochSecond();
            Audience newAudience = oldAudience.toBuilder()
                    .setTimestamps(Timestamps.newBuilder()
                            .setCreated(oldAudience.getTimestamps().getCreated())
                            .setModified(timestamp))
                    .setState(Audience.State.DELETING)
                    .build();
            tables.audiences().updateQuery(newAudience).execute();

            scheduleDeleteAudienceSegment(newAudience);
            return newAudience;
        });
    }

    @Override
    public Audience deleteLabAudience(String id) throws NotFoundException {
        return withAudienceForUpdate(id, (tables, segment) -> {
            Audience audience = fetchModifiableLabAudience(id, tables);
            tables.audiences().deleteQuery(id).execute();
            return audience;
        });
    }

    private void scheduleCreateAudienceSegment(Audience audience) {
        String id = audience.getId();
        QuartzScheduler quartz = schedulers.getQuartz();
        JobBuilder jobDetail = quartz.job(id, AudienceCreateJob.class);

        jobDetail = jobDetail.usingJobData(AudienceCreateJob.AUDIENCE_ID, id);
        jobDetail = jobDetail.usingJobData(AudienceCreateJob.LOGIN, audience.getLogin());
        jobDetail = jobDetail.usingJobData(AudienceCreateJob.SEGMENT_NAME, audience.getName());
        jobDetail = jobDetail.usingJobData(AudienceCreateJob.TABLE, audience.getSourcePath());
        jobDetail = jobDetail.usingJobData(AudienceCreateJob.TABLE_FIELD, audience.getSourceField());
        jobDetail = jobDetail.usingJobData(AudienceCreateJob.ENVIRONMENT, getSandboxEnvironment());

        quartz.schedule(jobDetail, quartz.startNow(id));
        LOG.info("Scheduled createAudienceSegment job {}", jobDetail);
    }

    private void scheduleModifyAudienceSegment(Audience audience) {
        String id = audience.getId();
        QuartzScheduler quartz = schedulers.getQuartz();
        JobBuilder jobDetail = quartz.job(id, AudienceModifyJob.class);

        jobDetail = jobDetail.usingJobData(AudienceModifyJob.AUDIENCE_ID, id);
        jobDetail = jobDetail.usingJobData(AudienceModifyJob.LOGIN, audience.getLogin());
        jobDetail = jobDetail.usingJobData(AudienceModifyJob.SEGMENT_ID, Long.toString(audience.getExternalId()));
        jobDetail = jobDetail.usingJobData(AudienceModifyJob.TABLE, audience.getSourcePath());
        jobDetail = jobDetail.usingJobData(AudienceModifyJob.TABLE_FIELD, audience.getSourceField());
        jobDetail = jobDetail.usingJobData(AudienceModifyJob.ENVIRONMENT, getSandboxEnvironment());

        quartz.schedule(jobDetail, quartz.startNow(id));
        LOG.info("Scheduled modifyAudienceSegment job {}", jobDetail);
    }

    private void scheduleDeleteAudienceSegment(Audience audience) {
        String id = audience.getId();
        QuartzScheduler quartz = schedulers.getQuartz();
        JobBuilder jobDetail = quartz.job(id, AudienceDeleteJob.class);

        jobDetail = jobDetail.usingJobData(AudienceDeleteJob.AUDIENCE_ID, id);
        jobDetail = jobDetail.usingJobData(AudienceDeleteJob.LOGIN, audience.getLogin());
        jobDetail = jobDetail.usingJobData(AudienceDeleteJob.SEGMENT_ID, Long.toString(audience.getExternalId()));
        jobDetail = jobDetail.usingJobData(AudienceDeleteJob.ENVIRONMENT, getSandboxEnvironment());

        quartz.schedule(jobDetail, quartz.startNow(id));
        LOG.info("Scheduled deleteAudienceSegment job {}", jobDetail);
    }

    @Override
    public DefaultAudienceService clone() {
        return new DefaultAudienceService(environment(), sql(), schedulers);
    }

}
