package ru.yandex.crypta.lab.yt;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;

import javax.inject.Inject;

import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.common.ws.EntityId;
import ru.yandex.crypta.lab.ConstructorService;
import ru.yandex.crypta.lab.base.BaseService;
import ru.yandex.crypta.lab.formatters.RuleFormatUtils;
import ru.yandex.crypta.lab.proto.Rule;
import ru.yandex.crypta.lab.proto.RuleCondition;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.Timestamps;
import ru.yandex.crypta.lab.rule_estimates.RuleEstimatorClient;
import ru.yandex.crypta.lab.rule_estimator.proto.RuleEstimateStats;
import ru.yandex.crypta.lab.tables.RulesConditionsTable;
import ru.yandex.crypta.lab.tables.Tables;
import ru.yandex.crypta.lab.utils.RuleApproval;
import ru.yandex.crypta.lib.proto.EEnvironment;

import static ru.yandex.crypta.lab.tables.RulesConditionsTable.REVISION;


public class DefaultConstructorService extends BaseService<ConstructorService> implements ConstructorService {
    private final RuleEstimatorClient ruleEstimatorClient;

    @Inject
    public DefaultConstructorService(EEnvironment environment, PostgresClient sql, RuleEstimatorClient ruleEstimatorClient) {
        super(environment, sql);
        this.ruleEstimatorClient = ruleEstimatorClient;
    }

    private <T> T withRuleConditionForUpdate(String ruleId, RuleCondition.Source source, RuleCondition.State state,
            BiFunction<Tables, RuleCondition, T> updater)
    {
        return withSqlTransaction(tables -> {
            RuleCondition condition = fetchModifiableRuleCondition(tables, ruleId, source, state);

            return updater.apply(tables, condition);
        });
    }

    private <T> T withRuleConditionsBySourceForUpdate(String ruleId, RuleCondition.Source source,
            BiFunction<Tables, List<RuleCondition>, T> updater)
    {
        return withSqlTransaction(tables -> {
            List<RuleCondition> conditions = fetchModifiableRuleConditionsBySource(tables, ruleId, source);

            return updater.apply(tables, conditions);
        });
    }

    private <T> T withRuleConditionsForUpdate(String ruleId, BiFunction<Tables, Rule.Builder, T> updater) {
        return withSqlTransaction(tables -> {
            Rule rule = fetchModifiableRule(tables, ruleId);

            return updater.apply(tables, rule.toBuilder());
        });
    }

    private Rule fetchRule(Tables tables, String id) {
        return tables
                .rules()
                .selectByIdQuery(id)
                .fetchOptionalInto(Rule.class)
                .orElseThrow(NotFoundException::new);
    }

    private Rule fetchModifiableRule(Tables tables, String id) {
        return tables
                .rules()
                .selectByIdModifiableQuery(id)
                .fetchOptionalInto(Rule.class)
                .orElseThrow(NotFoundException::new);
    }

    private Rule fetchAccessibleRule(Tables tables, String id) {
        return tables
                .rules()
                .selectByIdAccessibleQuery(id)
                .fetchOptionalInto(Rule.class)
                .orElseThrow(NotFoundException::new);
    }

    private Segment.Export fetchModifiableSegmentExport(Tables tables, String id) {
        return tables.segmentExports()
                .selectModifiableQuery(id)
                .fetchInto(Segment.Export.class).get(0);
    }

    @Override
    public List<Rule> getAllRules() {
        return tables()
                .rules()
                .selectQuery()
                .fetchInto(Rule.class);
    }

    private Rule buildRule(Rule.Builder prototype) {
        String id = new EntityId("rule").toString();

        long timestamp = Instant.now().getEpochSecond();
        return prototype
                .setId(id)
                .setAuthor(securityContext().getUserPrincipal().getName())
                .setTimestamps(Timestamps.newBuilder()
                        .setCreated(timestamp)
                        .setModified(timestamp))
                .build();
    }

    @Override
    public Rule createRule(Rule.Builder prototype) {
        return withSqlTransaction(tables -> {
            Rule rule = buildRule(prototype);
            tables.rules().insertQuery(rule).execute();
            return fetchRule(tables, rule.getId());
        });
    }

    @Override
    public Rule createExportRule(String exportId, Rule.Builder prototype) {
        return withSqlTransaction(tables -> {
            Segment.Export export = fetchModifiableSegmentExport(tables, exportId);
            Rule rule = buildRule(prototype);
            tables.rules().insertQuery(rule).execute();
            tables.segmentExports().updateRuleIdQuery(export.getId(), rule.getId()).execute();
            return fetchRule(tables, rule.getId());
        });
    }

    @Override
    public Rule getRule(String id) {
        return fetchAccessibleRule(tables(), id);
    }

    @Override
    public Rule updateRule(String id, Rule.Builder prototype) {
        return withSqlTransaction(tables -> {
            long timestamp = Instant.now().getEpochSecond();
            prototype.setTimestamps(Timestamps.newBuilder()
                    .setModified(timestamp)
            );

            Rule existing = fetchModifiableRule(tables, id);
            Rule updated = existing.toBuilder()
                    .mergeFrom(prototype.build())
                    .build();
            tables.rules().updateQuery(id, updated).execute();
            return fetchModifiableRule(tables, id);
        });
    }

    @Override
    public Rule deleteRule(String id) {
        return withRuleConditionsForUpdate(id, (tables, rule) -> {
            tables.rulesConditions().deleteByRuleIdQuery(id).execute();
            tables.rules().deleteQuery(id).execute();
            return rule.build();
        });
    }

    @Override
    public Rule deleteExportRule(String exportId) {
        Segment.Export export = fetchModifiableSegmentExport(tables(), exportId);
        String ruleId = export.getRuleId();
        return withRuleConditionsForUpdate(ruleId, (tables, rule) -> {
            tables.segmentExports().deleteRuleIdQuery(exportId).execute();
            tables.rulesConditions().deleteByRuleIdQuery(ruleId).execute();
            tables.rules().deleteQuery(ruleId).execute();
            return rule.build();
        });
    }

    private RuleCondition fetchAccessibleRuleCondition(Tables tables, String ruleId, RuleCondition.Source source,
            RuleCondition.State state)
    {
        return tables
                .rulesConditions()
                .selectAccessibleQuery(ruleId, source, Optional.of(state))
                .fetchOptionalInto(RuleCondition.class)
                .orElseThrow(NotFoundException::new);
    }

    private List<RuleCondition> fetchAccessibleRuleConditionsBySource(Tables tables, String ruleId,
            RuleCondition.Source source)
    {
        return tables
                .rulesConditions()
                .selectAccessibleQuery(ruleId, source)
                .fetchInto(RuleCondition.class);
    }

    private RuleCondition fetchAccessibleRuleConditionByRevision(Tables tables, Long revision) {
        return tables
                .rulesConditions()
                .selectByRevisionAccessibleQuery(revision)
                .fetchOptionalInto(RuleCondition.class)
                .orElseThrow(NotFoundException::new);
    }

    private Optional<RuleCondition> fetchOptionalModifiableRuleCondition(Tables tables, String ruleId,
            RuleCondition.Source source, RuleCondition.State state)
    {
        return tables
                .rulesConditions()
                .selectModifiableQuery(ruleId, source, state)
                .fetchOptionalInto(RuleCondition.class);
    }

    private RuleCondition fetchModifiableRuleCondition(Tables tables, String ruleId, RuleCondition.Source source,
            RuleCondition.State state)
    {
        return fetchOptionalModifiableRuleCondition(tables, ruleId, source, state)
                .orElseThrow(NotFoundException::new);
    }

    private List<RuleCondition> fetchModifiableRuleConditionsBySource(Tables tables, String ruleId,
            RuleCondition.Source source)
    {
        return tables
                .rulesConditions()
                .selectModifiableQuery(ruleId, source)
                .fetchInto(RuleCondition.class);
    }

    @Override
    public List<RuleCondition> getAllRulesConditions() {
        return tables()
                .rulesConditions()
                .selectQuery()
                .fetchInto(RuleCondition.class);
    }

    @Override
    public List<RuleCondition> getAllRulesConditionsBySource(RuleCondition.Source source) {
        return tables()
                .rulesConditions()
                .selectBySourceQuery(source)
                .fetchInto(RuleCondition.class);
    }

    @Override
    public RuleCondition putRuleCondition(String ruleId, RuleCondition.Source source, List<String> values) {
        return withSqlTransaction(tables -> {
            var formattedLines = RuleFormatUtils.parseValues(RulesConditionsTable.getFormatter(source), RuleFormatUtils.serializeValues(values));
            var fullValues = RuleFormatUtils.getFullValue(formattedLines);
            boolean hasErrors = RuleFormatUtils.hasErrors(fullValues);

            tables.segmentExports().updateRuleError(ruleId, hasErrors).execute();

            long timestamp = Instant.now().getEpochSecond();
            RuleCondition.State state = RuleApproval.getState(source, fullValues);

            Optional<RuleCondition> existing = fetchOptionalModifiableRuleCondition(tables, ruleId, source, state);
            if (existing.isPresent()) {
                tables.rulesConditions().updateValueQuery(ruleId, source, state, timestamp, values, hasErrors).execute();
            } else {
                tables.rulesConditions().insertQuery(ruleId, source, state, timestamp, values, hasErrors).execute();
            }

            updateRuleEstimateStats(ruleId, tables);

            return fetchModifiableRuleCondition(tables, ruleId, source, state);
        });
    }

    @Override
    public RuleCondition approveRuleCondition(String ruleId, RuleCondition.Source source) {
        return withSqlTransaction(tables -> {
            long timestamp = Instant.now().getEpochSecond();

            if (tables
                .rulesConditions()
                .selectApprovableQuery(ruleId, source, RuleCondition.State.NEED_APPROVE)
                .fetchOptionalInto(RuleCondition.class).isEmpty()
            ) {
                throw new NotFoundException();
            }

            Optional<RuleCondition> existing = fetchOptionalModifiableRuleCondition(tables, ruleId, source, RuleCondition.State.APPROVED);
            if (existing.isPresent()) {
                tables.rulesConditions().deleteQuery(ruleId, source, RuleCondition.State.APPROVED).execute();
            }

            tables.rulesConditions().approveQuery(ruleId, source, timestamp).execute();
            return fetchModifiableRuleCondition(tables, ruleId, source, RuleCondition.State.APPROVED);
        });
    }

    @Override
    public RuleCondition getRuleCondition(String ruleId, RuleCondition.Source source, RuleCondition.State state) {
        return fetchAccessibleRuleCondition(tables(), ruleId, source, state);
    }

    @Override
    public List<RuleCondition> getRuleConditionBySource(String ruleId, RuleCondition.Source source) {
        return fetchAccessibleRuleConditionsBySource(tables(), ruleId, source);
    }

    @Override
    public RuleCondition getRuleConditionByRevision(Long revision) {
        return fetchAccessibleRuleConditionByRevision(tables(), revision);
    }

    @Override
    public RuleCondition deleteRuleCondition(String ruleId, RuleCondition.Source source, RuleCondition.State state) {
        return withRuleConditionForUpdate(ruleId, source, state, (tables, existing) -> {
            tables().rulesConditions().deleteQuery(ruleId, source, state).execute();
            return existing;
        });
    }

    @Override
    public List<RuleCondition> deleteRuleConditionBySource(String ruleId, RuleCondition.Source source) {
        return withRuleConditionsBySourceForUpdate(ruleId, source, (tables, existing) -> {
            for (var condition: existing) {
                tables().rulesConditions().deleteQuery(condition.getRuleId(), condition.getSource(), condition.getState()).execute();
            }
            return existing;
        });
    }

    @Override
    public Optional<RuleEstimateStats> getRuleConditionEstimateStatsByRevision(Long revision) {
        return ruleEstimatorClient.getRuleConditionStats(revision);
    }

    @Override
    public Optional<RuleEstimateStats> getRuleEstimateStats(String ruleId) {
        return ruleEstimatorClient.getRuleStats(getRuleConditionIds(ruleId, tables()));
    }

    @Override
    public boolean updateRuleEstimateStats(String ruleId) {
        return updateRuleEstimateStats(ruleId, tables());
    }

    private boolean updateRuleEstimateStats(String ruleId, Tables tables) {
        return ruleEstimatorClient.update(ruleId, getRuleConditionIds(ruleId, tables)).isPresent();
    }

    private List<Long> getRuleConditionIds(String ruleId, Tables tables) {
        return tables.rulesConditions().selectQuery(ruleId).fetch(REVISION);
    }

    @Override
    public DefaultConstructorService clone() {
        return new DefaultConstructorService(environment(), sql(), ruleEstimatorClient);
    }
}
