package ru.yandex.crypta.lab.yt;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.inject.Inject;

import NCrypta.NSiberia.DescribingExperiment;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableList;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import okhttp3.MediaType;
import okhttp3.RequestBody;
import org.jooq.exception.DataAccessException;
import org.quartz.JobBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.audience.AudienceService;
import ru.yandex.crypta.audience.proto.TUserDataStats;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.common.ws.EntityId;
import ru.yandex.crypta.graph.soup.config.ProtobufEnum;
import ru.yandex.crypta.lab.CustomIdentifier;
import ru.yandex.crypta.lab.Identifier;
import ru.yandex.crypta.lab.SampleService;
import ru.yandex.crypta.lab.base.BaseYtService;
import ru.yandex.crypta.lab.job.DescribeSampleJob;
import ru.yandex.crypta.lab.job.DescribeSingleSampleJob;
import ru.yandex.crypta.lab.job.DescribeSubsamples;
import ru.yandex.crypta.lab.job.LearnSampleJob;
import ru.yandex.crypta.lab.job.LookalikeSampleJob;
import ru.yandex.crypta.lab.job.MatchSampleJob;
import ru.yandex.crypta.lab.job.PastDescribeSampleJob;
import ru.yandex.crypta.lab.job.UploadSampleToSiberiaJob;
import ru.yandex.crypta.lab.proto.AccessLevel;
import ru.yandex.crypta.lab.proto.EHashingMethod;
import ru.yandex.crypta.lab.proto.ELabIdentifierType;
import ru.yandex.crypta.lab.proto.ESampleViewState;
import ru.yandex.crypta.lab.proto.ESampleViewType;
import ru.yandex.crypta.lab.proto.Sample;
import ru.yandex.crypta.lab.proto.Subsamples;
import ru.yandex.crypta.lab.proto.TLookalikeOptions;
import ru.yandex.crypta.lab.proto.TMatchingOptions;
import ru.yandex.crypta.lab.proto.TSampleStats;
import ru.yandex.crypta.lab.proto.TSampleView;
import ru.yandex.crypta.lab.proto.TSampleViewOptions;
import ru.yandex.crypta.lab.proto.Timestamps;
import ru.yandex.crypta.lab.siberia.RawResponse;
import ru.yandex.crypta.lab.siberia.SiberiaClient;
import ru.yandex.crypta.lab.tables.Tables;
import ru.yandex.crypta.lab.utils.Acl;
import ru.yandex.crypta.lab.utils.Paths;
import ru.yandex.crypta.lab.utils.SchemaField;
import ru.yandex.crypta.lab.utils.YtParameters;
import ru.yandex.crypta.lab.utils.YtQueries;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.IdentifiersEnumExtensions;
import ru.yandex.crypta.lib.proto.identifiers.TIdType;
import ru.yandex.crypta.lib.proto.identifiers.TIds;
import ru.yandex.crypta.lib.schedulers.JobView;
import ru.yandex.crypta.lib.schedulers.QuartzScheduler;
import ru.yandex.crypta.lib.schedulers.Schedulers;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.siberia.core.proto.TAddUserSetResponse;
import ru.yandex.crypta.siberia.proto.TDescribeIdsResponse;
import ru.yandex.crypta.siberia.proto.TStats;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;


public class DefaultSampleService extends BaseYtService<SampleService> implements SampleService {

    public static final String ROW_COUNT = "row_count";
    public static final String SCHEMA = "schema";
    public static final String IDENTITY = "identity";
    public static final String SOURCE_VIEW_ID = "source";
    private static final String YANDEXUIDS = "yandexuid";
    private static final String INVALID = "invalid";
    private static final String ID_PREFIX = "sample";
    private static final String VIEW_PREFIX = "view";
    private static final String IDENTITY_VIEW_ID = "identity";
    private static final String YANDEXUID_VIEW_ID = "yandexuid";
    private static final String INVALID_VIEW_ID = "invalid";
    private static final long MAX_SIBERIA_USER_SET_ROW_COUNT = 50 * 1000 * 1000;

    private static final Set<ESampleViewType> DEFAULT_VIEW_TYPES = Cf.set(
            ESampleViewType.IDENTITY,
            ESampleViewType.SOURCE,
            ESampleViewType.YANDEXUID,
            ESampleViewType.INVALID
    );

    private static final ProtobufEnum<EIdType, TIdType> ID_TYPES = new ProtobufEnum<>(
            EIdType.class,
            EIdType.getDescriptor(),
            TIdType.getDescriptor(),
            TIdType.newBuilder(),
            ImmutableList.of(
                    IdentifiersEnumExtensions.name,
                    IdentifiersEnumExtensions.repObj
            )
    );

    private static final Logger LOG = LoggerFactory.getLogger(DefaultSampleService.class);
    public static final MediaType APPLICATION_JSON = MediaType.get("application/json");
    private final Paths paths;
    private final Schedulers schedulers;
    private final AudienceService audience;
    private final SiberiaClient siberiaClient;

    private final Cache<String, List<String>> groupsCache = CacheBuilder.newBuilder()
            .maximumSize(100)
            .expireAfterWrite(1, TimeUnit.HOURS)
            .build();

    @Inject
    public DefaultSampleService(
            EEnvironment environment,
            PostgresClient sql,
            YtService yt,
            Schedulers schedulers,
            AudienceService audience,
            SiberiaClient siberiaClient
    )
    {
        super(environment, sql, yt);
        this.paths = DefaultLabService.getPaths(environment);
        this.schedulers = schedulers;
        this.audience = audience;
        this.siberiaClient = siberiaClient;
    }

    private static void validateTable(YTreeNode table) {
        validateSize(table.getAttributes());
    }

    private static boolean isEmptyString(String data) {
        return "".equals(data);
    }

    static SchemaField validateSchemaAndObtainIdField(YTreeNode table, CustomIdentifier customIdentifier, String dateKey) {
        List<SchemaField> fields = Cf.wrap(table.getAttributes())
                .getOptional(SCHEMA)
                .orElseThrow(() -> Exceptions.wrongRequestException("No schema in table", "MISSED_SCHEMA"))
                .asList()
                .stream()
                .map(YTreeNode::asMap)
                .map(SchemaField::fromMap)
                .collect(Collectors.toList());

        Optional<SchemaField> identifierField = fields.stream().filter(customIdentifier::matches).findFirst();
        if (!identifierField.isPresent()) {
            throw Exceptions
                    .wrongRequestException("No identifier found: " + customIdentifier, "MISSED_OR_WRONG_IDENTIFIER");
        }
        if (dateKey != null) {
            Optional<SchemaField> dateField = fields.stream().filter(field -> Objects.equals(dateKey, field.getName())).findFirst();
            if (!dateField.isPresent()) {
                throw Exceptions
                        .wrongRequestException("No date field found: " + dateKey, "MISSED_OR_WRONG_DATE_FIELD");
            }
        }
        return identifierField.get();
    }

    public static boolean validateSize(Map<String, YTreeNode> tableAttributes) {
        if (tableAttributes.get(ROW_COUNT).longValue() == 0) {
            throw Exceptions.wrongRequestException("Empty table", "TABLE_IS_EMPTY");
        }
        return true;
    }

    @Override
    public Sample getSample(String id) throws NotFoundException {
        return fetchSample(tables(), id);
    }

    private TSampleView createIdentityView(TSampleView sourceView, String path) {
        TSampleView.Builder builder = TSampleView.newBuilder()
                .setID(IDENTITY_VIEW_ID)
                .setPath(path)
                .setSampleID(sourceView.getSampleID())
                .setState(ESampleViewState.READY)
                .setType(ESampleViewType.IDENTITY);
        TMatchingOptions sourceMatchingOptions = sourceView.getOptions().getMatching();
        builder.getOptionsBuilder()
                .setDerivedFrom(sourceView.getID())
                .getMatchingBuilder()
                .setKey(sourceMatchingOptions.getKey())
                .setHashingMethod(sourceMatchingOptions.getHashingMethod())
                .setIncludeOriginal(true)
                .setIdType(sourceMatchingOptions.getIdType());
        return builder.build();
    }

    private TSampleView createInvalidView(TSampleView identityView, String path) {
        TSampleView.Builder builder = TSampleView.newBuilder()
                .setID(INVALID_VIEW_ID)
                .setPath(path)
                .setSampleID(identityView.getSampleID())
                .setState(ESampleViewState.NEW)
                .setType(ESampleViewType.INVALID);
        TMatchingOptions sourceMatchingOptions = identityView.getOptions().getMatching();
        builder.getOptionsBuilder()
                .setDerivedFrom(identityView.getID())
                .getMatchingBuilder()
                .setKey(sourceMatchingOptions.getKey())
                .setHashingMethod(sourceMatchingOptions.getHashingMethod())
                .setIncludeOriginal(true)
                .setIdType(sourceMatchingOptions.getIdType());
        return builder.build();
    }

    private TSampleView createYandexuidView(TSampleView identityView, String path) {
        TSampleView.Builder builder = TSampleView.newBuilder()
                .setID(YANDEXUID_VIEW_ID)
                .setPath(path)
                .setSampleID(identityView.getSampleID())
                .setState(ESampleViewState.NEW)
                .setType(ESampleViewType.YANDEXUID);
        builder.getOptionsBuilder()
                .setDerivedFrom(identityView.getID())
                .getMatchingBuilder()
                .setKey("yuid")
                .setHashingMethod(EHashingMethod.HM_IDENTITY)
                .setIncludeOriginal(true)
                .setIdType(ELabIdentifierType.LAB_ID_YANDEXUID);
        return builder.build();
    }

    private YTreeListNode getDirectoryAcl(List<YTreeNode> originalAcl, AccessLevel accessLevel) {
        if (Objects.equals(accessLevel, AccessLevel.PUBLIC)) {
            return Acl.publicAcl();
        } else {
            return Acl.readOnly(originalAcl);
        }
    }

    @Override
    public Sample createSample(YPath path, String name, CustomIdentifier customIdentifier, String dateKey,
            String groupingKey, Long ttl, AccessLevel accessLevel, Integer maxGroupsCount)
    {
        return withSqlTransaction(tables -> {
            var resultSample = withYtTransaction(ytTransaction -> {
                String id = new EntityId(ID_PREFIX).toString();
                List<YTreeNode> acl = Acl.get(cypress(), optionalId(ytTransaction), path);
                YPath destinationRoot = paths.sample(id);
                createDirectory(ytTransaction, destinationRoot, getDirectoryAcl(acl, accessLevel));
                YPath destinationIdentity = destinationRoot.child(IDENTITY);

                copyTable(ytTransaction, path, destinationIdentity);

                YTreeNode table = adoptableTable(destinationIdentity, ytTransaction);
                validateTable(table);
                SchemaField idField = validateSchemaAndObtainIdField(table, customIdentifier, dateKey);

                if (Objects.equals(idField.getName(), groupingKey)) {
                    throw Exceptions
                            .wrongRequestException("Grouping key can't be equal to id key", "GROUPING_KEY_EQUAL_TO_ID");
                }

                Sample.Builder sample = Sample.newBuilder()
                        .setId(id)
                        .setAuthor(getLogin())
                        .setName(name)
                        .setIdName(customIdentifier.getIdentifier().getName())
                        .setIdKey(idField.getName())
                        .setDateKey(MoreObjects.firstNonNull(dateKey, ""))
                        .setGroupingKey(MoreObjects.firstNonNull(groupingKey, ""))
                        .setTtl(ttl)
                        .setAccessLevel(MoreObjects.firstNonNull(accessLevel, AccessLevel.PRIVATE))
                        .setMaxGroupsCount(maxGroupsCount)
                        .setState("CREATED");

                sample.getTimestampsBuilder().mergeFrom(getInitialTimestamps());

                TSampleView.Builder sourceView = TSampleView.newBuilder()
                        .setID(SOURCE_VIEW_ID)
                        .setPath(path.toString())
                        .setSampleID(id)
                        .setState(ESampleViewState.READY)
                        .setType(ESampleViewType.SOURCE);
                sourceView.getOptionsBuilder()
                        .getMatchingBuilder()
                        .setKey(sample.getIdKey())
                        .setHashingMethod(customIdentifier.getIdentifier().getHashingMethod())
                        .setIncludeOriginal(true)
                        .setIdType(customIdentifier.getIdentifier().getLabType());

                tables.samples().insertQuery(sample.build()).execute();

                TSampleView identityView = createIdentityView(sourceView.build(), destinationIdentity.toString());
                tables.sampleViews().insertQuery(identityView).execute();
                tables.sampleViews().insertQuery(sourceView.build()).execute();

                var isRegularSample = isEmptyString(sample.getDateKey());

                if (isRegularSample) {
                    TSampleView yandexuidView = createYandexuidView(identityView, destinationRoot.child(YANDEXUIDS).toString());
                    TSampleView invalidView = createInvalidView(identityView, destinationRoot.child(INVALID).toString());

                    tables.sampleViews().insertQuery(yandexuidView).execute();
                    tables.sampleViews().insertQuery(invalidView).execute();

                    describe(sample.build(), identityView, yandexuidView, invalidView);
                } else {
                    schedulePastDescribeJob(sample.build(), identityView);
                }

                return fetchSample(tables, id);
            });

            tryScheduleUploadToSiberiaJob(resultSample, customIdentifier);

            return resultSample;
        });
    }

    @Override
    public Sample setUserSetId(String id, String userSetId) {
        return withSqlTransaction(tables -> {
            Sample.Builder sample = getSample(id).toBuilder();
            sample.setSiberiaUserSetId(userSetId);

            tables().samples().updateQuery(sample.build()).execute();

            return getSample(id);
        });
    }

    private long getNowTimestamp() {
        return Instant.now().getEpochSecond();
    }

    private Timestamps getInitialTimestamps() {
        long now = getNowTimestamp();
        return Timestamps.newBuilder().setCreated(now).setModified(now).build();
    }

    private Sample fetchAllowedSample(Tables tables, String id) {
        Sample sample = fetchSample(tables, id);

        if (!Acl.canCreateViewsForSample(sample, securityContext())) {
            throw Exceptions.forbidden("You can't create views for " + sample.getId(),
                    "CANNOT_CREATE_VIEW_FOR_THAT_SAMPLE");
        }

        return sample;
    }

    private TSampleView fetchIdentityView(Tables tables, String id) {
        return tables
                .sampleViews()
                .selectByIdQuery(id, IDENTITY_VIEW_ID)
                .fetchOptionalInto(TSampleView.class)
                .orElseThrow(
                        () -> Exceptions.wrongRequestException("Sample has no identity view", "SAMPLE_IS_BROKEN"));
    }

    private TSampleView fetchYandexuidView(Tables tables, String id) {
        return tables
                .sampleViews()
                .selectByIdQuery(id, YANDEXUID_VIEW_ID)
                .fetchOptionalInto(TSampleView.class)
                .orElseThrow(
                        () -> Exceptions.wrongRequestException("Sample has no yandexuid view", "SAMPLE_IS_BROKEN"));
    }

    @Override
    public TSampleView createMatchingView(String id, TSampleViewOptions options) throws NotFoundException {
        if (!Acl.canCreateSuchView(options.getMatching(), securityContext())) {
            throw Exceptions.forbidden("You don't have role for such view", "MISSED_ROLE_FOR_SUCH_VIEW");
        }

        return withSqlTransaction(tables -> {
            var sample = fetchAllowedSample(tables, id);
            var identityView = fetchIdentityView(tables, id);

            String viewId = new EntityId(VIEW_PREFIX).toString();
            YPath viewPath =
                    safePath(identityView.getPath()).orElseThrow(IllegalArgumentException::new).parent().child(viewId);

            TSampleView.Builder view = TSampleView.newBuilder()
                    .setID(viewId)
                    .setPath(viewPath.toString())
                    .setSampleID(sample.getId())
                    .setState(ESampleViewState.NEW)
                    .setType(ESampleViewType.MATCHING);

            if (options.getMatching().getIdType().equals(identityView.getOptions().getMatching().getIdType())) {
                throw Exceptions.wrongRequestException("Identity view has equal id type.", "EQUAL_ID_TYPE");
            }

            view.getOptionsBuilder()
                    .setDerivedFrom(identityView.getID())
                    .getMatchingBuilder()
                    .mergeFrom(options.getMatching())
                    .setKey(MoreObjects.firstNonNull(
                            Strings.emptyToNull(options.getMatching().getKey()),
                            Identifier.byLabType(options.getMatching().getIdType()).getName()
                    ));

            tables.sampleViews().insertOrUpdateQuery(view.build()).execute();

            scheduleMatchingJob(sample, identityView, view.build());

            return view.build();
        });
    }

    @Override
    public TSampleView createCryptaIdStatisticsView(String id) {
        TMatchingOptions.Builder options = TMatchingOptions.newBuilder()
                .setIncludeOriginal(true)
                .setKey("counts");

        return withSqlTransaction(tables -> {
            var sample = fetchAllowedSample(tables, id);
            var identityView = fetchIdentityView(tables, id);

            String viewId = new EntityId(VIEW_PREFIX).toString();
            YPath viewPath =
                    safePath(identityView.getPath()).orElseThrow(IllegalArgumentException::new).parent().child(viewId);

            TSampleView.Builder view = TSampleView.newBuilder()
                    .setID(viewId)
                    .setPath(viewPath.toString())
                    .setSampleID(sample.getId())
                    .setState(ESampleViewState.NEW)
                    .setType(ESampleViewType.CRYPTA_ID_STATISTICS);

            view.getOptionsBuilder()
                    .setDerivedFrom(identityView.getID())
                    .setMatching(options);

            tables.sampleViews().insertOrUpdateQuery(view.build()).execute();

            scheduleMatchingJob(sample, identityView, view.build());

            return view.build();
        });
    }

    @Override
    public TSampleView createLookalikeView(String id, int outputCount, boolean useDates) {
        TLookalikeOptions.TCounts.Builder countsOptions = TLookalikeOptions.TCounts.newBuilder()
                .setOutput(outputCount);
        TLookalikeOptions.Builder options = TLookalikeOptions.newBuilder()
                .setCounts(countsOptions)
                .setUseDates(useDates);

        return withSqlTransaction(tables -> {
            var sample = fetchAllowedSample(tables, id);
            var identityView = fetchIdentityView(tables, id);
            var srcView = identityView;
            if (!useDates) {
                srcView = fetchYandexuidView(tables, id);
            }

            String viewId = new EntityId(VIEW_PREFIX).toString();
            YPath viewPath =
                    safePath(identityView.getPath()).orElseThrow(IllegalArgumentException::new).parent().child(viewId);

            TSampleView.Builder view = TSampleView.newBuilder()
                    .setID(viewId)
                    .setPath(viewPath.toString())
                    .setSampleID(sample.getId())
                    .setState(ESampleViewState.NEW)
                    .setType(ESampleViewType.LOOKALIKE);

            view.getOptionsBuilder()
                    .setDerivedFrom(srcView.getID())
                    .setLookalike(options.build());

            tables.sampleViews().insertOrUpdateQuery(view.build()).execute();

            scheduleLookalikeJob(sample, srcView, view.build());

            return view.build();
        });
    }

    @Override
    public boolean canCreateView(String id, TMatchingOptions options) throws NotFoundException {
        return withSqlTransaction(tables -> {
            Sample sample = fetchSample(tables, id);

            if (!Acl.canCreateViewsForSample(sample, securityContext())) {
                return false;
            }

            if (!Acl.canCreateSuchView(options, securityContext())) {
                return false;
            }

            return true;
        });
    }

    private TSampleView fetchView(Tables tables, String sampleId, String viewId) {
        return tables.sampleViews().selectByIdQuery(sampleId, viewId).fetchOneInto(TSampleView.class);
    }

    private void scheduleLookalikeJob(Sample sample, TSampleView source, TSampleView destination) {
        withQuartz(quartz -> {
            String sampleId = sample.getId();
            JobBuilder jobDetail = quartz.job(sampleId, LookalikeSampleJob.class)
                    .usingJobData(LookalikeSampleJob.SAMPLE_ID, sampleId)
                    .usingJobData(LookalikeSampleJob.SRC_VIEW, source.getID())
                    .usingJobData(LookalikeSampleJob.DST_VIEW, destination.getID())
                    .usingJobData(LookalikeSampleJob.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(sampleId));
            return jobDetail;
        });
    }

    private void scheduleMatchingJob(Sample sample, TSampleView source, TSampleView destination) {
        withQuartz(quartz -> {
            String sampleId = sample.getId();
            JobBuilder jobDetail = quartz.job(sampleId, MatchSampleJob.class)
                    .usingJobData(MatchSampleJob.SAMPLE_ID, sampleId)
                    .usingJobData(MatchSampleJob.SRC_VIEW, source.getID())
                    .usingJobData(MatchSampleJob.DST_VIEW, destination.getID())
                    .usingJobData(MatchSampleJob.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(sampleId));
            return jobDetail;
        });
    }

    private String scheduleDescribeJob(
            Sample sample, TSampleView identity, TSampleView yandexuid, TSampleView invalid
    )
    {
        return withQuartz(quartz -> {
            String sampleId = sample.getId();
            JobBuilder jobDetail = quartz.job(sampleId, DescribeSampleJob.class)
                    .usingJobData(DescribeSampleJob.SAMPLE_ID, sampleId)
                    .usingJobData(DescribeSampleJob.VIEW_ID, identity.getID())
                    .usingJobData(DescribeSampleJob.YANDEXUID_VIEW_ID, yandexuid.getID())
                    .usingJobData(DescribeSampleJob.INVALID_VIEW_ID, invalid.getID())
                    .usingJobData(DescribeSampleJob.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(sampleId));
            LOG.info("Scheduled describe job {}", jobDetail);
            return sampleId;
        });
    }

    private String schedulePastDescribeJob(Sample sample, TSampleView identity)
    {
        return withQuartz(quartz -> {
            String sampleId = sample.getId();
            JobBuilder jobDetail = quartz.job(sampleId, PastDescribeSampleJob.class)
                    .usingJobData(PastDescribeSampleJob.SAMPLE_ID, sampleId)
                    .usingJobData(PastDescribeSampleJob.VIEW_ID, identity.getID())
                    .usingJobData(PastDescribeSampleJob.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(sampleId));
            LOG.info("Scheduled past describe job {}", jobDetail);
            return sampleId;
        });
    }

    private String scheduleDescribeSingleSampleJob(Sample sample, TSampleView identity) {
        return withQuartz(quartz -> {
            String sampleId = sample.getId();
            JobBuilder jobDetail = quartz.job(sampleId, DescribeSingleSampleJob.class)
                    .usingJobData(DescribeSingleSampleJob.SAMPLE_ID, sampleId)
                    .usingJobData(DescribeSingleSampleJob.VIEW_ID, identity.getID())
                    .usingJobData(DescribeSingleSampleJob.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(sampleId));
            LOG.info("Scheduled describe in Siberia job {}: sampleId={}", jobDetail, sampleId);
            return sampleId;
        });
    }

    private String scheduleDescribeSubsamplesJob(Sample sample, TSampleView identity) {
        return withQuartz(quartz -> {
            String sampleId = sample.getId();
            JobBuilder jobDetail = quartz.job(sampleId, DescribeSubsamples.class)
                    .usingJobData(DescribeSubsamples.SAMPLE_ID, sampleId)
                    .usingJobData(DescribeSubsamples.VIEW_ID, identity.getID())
                    .usingJobData(DescribeSubsamples.MAX_GROUPS_COUNT, String.valueOf(sample.getMaxGroupsCount()))
                    .usingJobData(DescribeSubsamples.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(sampleId));
            LOG.info("Scheduled describe in Siberia job {}", jobDetail);
            return sampleId;
        });
    }

    private void scheduleLearnJob(Sample sample) {
        withQuartz(quartz -> {
            String sampleId = sample.getId();
            JobBuilder jobDetail = quartz.job(sampleId, LearnSampleJob.class)
                    .usingJobData(LearnSampleJob.SAMPLE_ID, sampleId)
                    .usingJobData(LearnSampleJob.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(sampleId));
            return jobDetail;
        });
    }

    private void scheduleUploadToSiberiaJob(YPath path, String userSetId, String login,
            CustomIdentifier customIdentifier, String sampleId)
    {
        withQuartz(quartz -> {
            String idType = ID_TYPES.getByType(customIdentifier.getIdentifier().getIdType()).getName();
            JobBuilder jobDetail = quartz.job(userSetId, UploadSampleToSiberiaJob.class)
                    .usingJobData(UploadSampleToSiberiaJob.PATH, path.toString())
                    .usingJobData(UploadSampleToSiberiaJob.USER_SET_ID, userSetId)
                    .usingJobData(UploadSampleToSiberiaJob.SAMPLE_ID, sampleId)
                    .usingJobData(UploadSampleToSiberiaJob.LOGIN, login)
                    .usingJobData(UploadSampleToSiberiaJob.ID_FIELD, customIdentifier.getName())
                    .usingJobData(UploadSampleToSiberiaJob.ID_TYPE, idType)
                    .usingJobData(UploadSampleToSiberiaJob.ENVIRONMENT, getSandboxEnvironment());

            quartz.schedule(jobDetail, quartz.startNow(userSetId));
            LOG.info("Scheduled upload to Siberia job: {}", jobDetail);
            return jobDetail;
        });
    }

    private <T> T withQuartz(Function<QuartzScheduler, T> callback) {
        return callback.apply(schedulers.getQuartz());
    }

    @Override
    public Optional<TUserDataStats> getStats(String sampleId, String groupId) throws NotFoundException {
        Sample sample = getSample(sampleId);

        switch (sample.getType()) {
            case REGULAR:
                return getStatsRegular(sample, groupId);
            case AUDIENCE:
                return getStatsAudience(groupId);
            default:
                throw Exceptions.illegal("Wrong sample type");
        }
    }

    private Optional<TUserDataStats> getStatsRegular(Sample sample, String groupId) {
        // if yandexuids are broken then stats is broken too
        if (isEmptyString(sample.getDateKey())) {
            TSampleView view = getView(sample.getId(), YANDEXUID_VIEW_ID);
            if (view.getState().equals(ESampleViewState.ERROR)) {
                throw Exceptions.wrongRequestException("Failed to compute yandexuid view", "NO_STATS");
            }
        }

        List<YTreeMapNode> results = new ArrayList<>();
        yt().tables().selectRows(
                YtQueries.selectSingleSampleStats(paths.sampleStats().toString(), sample.getId(), groupId),
                YTableEntryTypes.YSON,
                (Consumer<YTreeMapNode>) results::add
        );
        if (results.isEmpty()) {
            return Optional.empty();
        }
        try {
            return Optional.of(TUserDataStats.parseFrom(results.get(0).getBytes("Stats")));
        } catch (InvalidProtocolBufferException e) {
            throw Exceptions.illegal(e.getMessage());
        }
    }

    private Optional<TUserDataStats> getStatsAudience(String groupId) {
        if (groupId.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(audience.getStats("audience", groupId));
    }

    private String createSiberiaUserSet(String title, Long ttl) {
        var response = siberiaClient.tryCall(siberia -> siberia.createUserSet(title, ttl));

        var responseCode = response.getHttpCode();
        if (responseCode != 200) {
            throw Exceptions.internal(
                    "user_sets/add Siberia handle return code = '" + responseCode + "' and message = '"
                            + response.getText() + "'");
        }

        try {
            var builder = TAddUserSetResponse.newBuilder();
            JsonFormat.parser().merge(response.getText(), builder);
            return builder.build().getUserSetId();
        } catch (IOException e) {
            throw Exceptions.internal("Error while parsing response from Siberia: " + e.getMessage());
        }
    }

    @Override
    public String uploadSampleToSiberia(YPath path, String title, CustomIdentifier customIdentifier, Long ttl) {
        return uploadSampleToSiberia(path, title, customIdentifier, "no_sample_id", ttl);
    }

    private void validateTableForSiberiaUserSet(YTreeNode table) {
        if (table.getAttributeOrThrow(ROW_COUNT).longValue() > MAX_SIBERIA_USER_SET_ROW_COUNT) {
            throw Exceptions.wrongRequestException("Too big table", "TABLE_IS_TOO_BIG");
        }
    }

    private void tryScheduleUploadToSiberiaJob(Sample sample, CustomIdentifier customIdentifier) {
        var siberiaUserSetId = sample.getSiberiaUserSetId();
        if (!siberiaUserSetId.isEmpty()) {
            var sampleId = sample.getId();
            YPath path = paths.sample(sampleId).child(IDENTITY);
            try {
                scheduleUploadToSiberiaJob(path, siberiaUserSetId, getLogin(), customIdentifier, sampleId);
            } catch (Exception e) {
                LOG.error("Error while scheduling upload to SIberia", e);
            }
        }
    }

    public String uploadSampleToSiberia(YPath path, String title, CustomIdentifier customIdentifier, String sampleId,
            Long ttl)
    {
        withYtTransaction(ytTransaction -> {
            validateTableForSiberiaUserSet(adoptableTable(path, ytTransaction));
            return true;
        });
        var login = getLogin();
        var siberiaUserSetId = createSiberiaUserSet(title, ttl);
        scheduleUploadToSiberiaJob(path, siberiaUserSetId, login, customIdentifier, sampleId);
        return siberiaUserSetId;
    }

    @Override
    public Sample updateSampleState(String id, String state) {
        return withSqlTransaction(tables -> {
            tables.samples().updateSampleStateQuery(id, state).execute();
            return fetchSample(tables, id);
        });
    }

    @Override
    public Sample updateSampleSource(String id, String source) {
        return withSqlTransaction(tables -> withYtTransaction(ytTransaction -> {
            Sample sample = getSample(id);
            var isRegularSample = isEmptyString(sample.getDateKey());
            ListF<TSampleView> views = Cf.x(fetchViews(tables(), id));
            Function<String, Optional<TSampleView>> findView = (viewId) ->
                    views.find(x -> Objects.equals(viewId, x.getID())).toOptional();

            TSampleView sourceView = findView.apply(SOURCE_VIEW_ID)
                    .map(view -> view.toBuilder().setPath(source).build())
                    .orElseThrow(() -> Exceptions.illegal("Sample has no source"));

            TSampleView identityView = findView.apply(IDENTITY_VIEW_ID)
                    .map(view -> view.toBuilder().setState(ESampleViewState.READY).build())
                    .orElseGet(() -> createIdentityView(sourceView, paths.sample(id).child(IDENTITY).toString()));

            YPath sourcePath = YPath.simple(source);

            List<YTreeNode> acl = Acl.get(cypress(), optionalId(ytTransaction), sourcePath);
            YTreeListNode readOnlyAcl = Acl.readOnly(acl);
            YPath destinationPath = safePath(identityView.getPath())
                    .orElseThrow(() -> Exceptions.illegal("Sample has invalid identity view"));
            createDirectory(ytTransaction, destinationPath.parent(), readOnlyAcl);
            YTreeNode table = adoptableTable(sourcePath, ytTransaction);

            copyTable(ytTransaction, sourcePath, destinationPath);
            validateTable(table);

            sample = sample.toBuilder()
                    .setTimestamps(getTimestampsModifiedNow(sample))
                    .build();

            tables.samples().updateQuery(sample).execute();
            tables.sampleViews().insertOrUpdateQuery(sourceView).execute();
            tables.sampleViews().insertOrUpdateQuery(identityView).execute();

            TSampleView yandexuidView;
            if (isRegularSample) {
                 yandexuidView = findView.apply(YANDEXUID_VIEW_ID)
                        .map(view -> view.toBuilder().setState(ESampleViewState.NEW).build())
                        .orElseGet(() -> createYandexuidView(identityView, paths.sample(id).child(YANDEXUIDS).toString()));

                TSampleView invalidView = findView.apply(INVALID_VIEW_ID)
                        .map(view -> view.toBuilder().setState(ESampleViewState.READY).build())
                        .orElseGet(() -> createInvalidView(identityView, paths.sample(id).child(INVALID).toString()));

                tables.sampleViews().insertOrUpdateQuery(yandexuidView).execute();
                tables.sampleViews().insertOrUpdateQuery(invalidView).execute();

                describe(sample, identityView, yandexuidView, invalidView);
            } else {
                yandexuidView = null;
                schedulePastDescribeJob(sample, identityView);
            }

            Sample updatedSample = fetchSample(tables, id);

            views.filterNot(each -> DEFAULT_VIEW_TYPES.contains(each.getType()))
                    .forEach(each -> {
                        var viewType = each.getType();

                        if (
                                viewType == ESampleViewType.MATCHING ||
                                viewType == ESampleViewType.CRYPTA_ID_STATISTICS
                        ) {
                            scheduleMatchingJob(updatedSample, identityView, each);
                        } else if (viewType == ESampleViewType.LOOKALIKE) {
                            if (isRegularSample) {
                                scheduleLookalikeJob(updatedSample, yandexuidView, each);
                            } else {
                                scheduleLookalikeJob(updatedSample, identityView, each);
                            }
                        } else {
                            throw Exceptions.unsupported();
                        }
                    });

            return updatedSample;
        }));
    }

    private Timestamps getTimestampsModifiedNow(Sample sample) {
        return sample.toBuilder()
                .getTimestampsBuilder()
                .setModified(getNowTimestamp())
                .build();
    }

    @Override
    public Sample deleteSample(String id) throws NotFoundException {
        return withSqlTransaction(tables -> withYtTransaction(ytTransaction -> {
            Sample sample = fetchSample(tables, id);
            List<TSampleView> views = fetchViews(tables, sample.getId());

            if (!Objects.equals(sample.getAuthor(), getLogin())) {
                throw Exceptions.illegal("Can't delete sample of someone else");
            }

            views.forEach(each -> {
                // TODO ouch that might be dangerous
                if (!Objects.equals(each.getID(), SOURCE_VIEW_ID)) {
                    deleteTableIfExists(ytTransaction, each.getPath());
                }
            });
            tables.sampleViews().deleteBySampleIdQuery(sample.getId()).execute();
            tables.samples().deleteByIdQuery(sample.getId()).execute();
            return sample;
        }));
    }

    private List<TSampleView> fetchViews(Tables tables, String sampleId) {
        return tables
                .sampleViews()
                .selectBySampleIdQuery(sampleId)
                .fetchInto(TSampleView.class);
    }

    private void describe(Sample sample, TSampleView identityView, TSampleView yandexuidView, TSampleView invalidView) {
        // TODO(unretrofied): schedule CreateStandardViews when old Describe job dropped
        // scheduleCreateStandardViewsJob(sample, identityView, yandexuidView, invalidView);
        scheduleDescribeJob(sample, identityView, yandexuidView, invalidView);

        // TODO(unretrofied): schedule CreateStandardViews when old Describe job dropped
        if (!isEmptyString(sample.getGroupingKey())) {
            // TODO(unretrofied): unify description when old Describe job dropped
            scheduleDescribeSubsamplesJob(sample, identityView);
        } else {
            scheduleDescribeSingleSampleJob(sample, identityView);
        }
    }

    @Override
    public Sample describe(String id) throws NotFoundException {
        return withSqlTransaction(tables -> {
            Sample sample = fetchSample(tables, id);
            TSampleView identityView = getView(id, IDENTITY_VIEW_ID);
            TSampleView yandexuidView = getView(id, YANDEXUID_VIEW_ID);
            TSampleView invalidView = getView(id, INVALID_VIEW_ID);
            describe(sample, identityView, yandexuidView, invalidView);
            tables.samples().updateQuery(sample).execute();
            return sample;
        });
    }

    @Override
    public Sample pastDescribe(String id) throws NotFoundException {
        return withSqlTransaction(tables -> {
            Sample sample = fetchSample(tables, id);
            TSampleView identityView = getView(id, IDENTITY_VIEW_ID);
            schedulePastDescribeJob(sample, identityView);
            tables.samples().updateQuery(sample).execute();
            return sample;
        });
    }

    @Override
    public Sample learn(String sampleId) {
        return withSqlTransaction(tables -> {
            Sample sample = fetchSample(tables, sampleId);
            scheduleLearnJob(sample);
            return sample;
        });
    }

    @Override
    public List<TSampleView> deleteOutdatedSampleViews() {
        return withLongSqlTransaction(tables -> withLongYtTransaction(ytTransaction -> {
            Long now = getNowTimestamp();

            List<TSampleView> outdatedViews = tables.sampleViews()
                    .selectOutdatedQuery(now)
                    .fetchInto(TSampleView.class);

            outdatedViews.forEach(each -> {
                if (!Objects.equals(each.getID(), SOURCE_VIEW_ID)) {
                    tables.sampleViews()
                            .updateStateQuery(each.getSampleID(), each.getID(), ESampleViewState.DELETED)
                            .execute();
                    deleteTableIfExists(ytTransaction, each.getPath());
                }
            });

            LOG.info("Outdated views deleted: " + outdatedViews.size());
            return outdatedViews;
        }));
    }

    @Override
    public List<String> deleteOutdatedSamples() {
        return withLongSqlTransaction(tables -> withLongYtTransaction(ytTransaction -> {
            Long now = getNowTimestamp();

            var outdatedSampleIds = tables.samples()
                    .selectOutdatedQuery(now)
                    .fetchInto(Sample.class).stream()
                    .map(Sample::getId)
                    .collect(Collectors.toSet());

            var samplesToDelete = cypress().list(Optional.of(ytTransaction.getId()), YtParameters.DO_NOT_PING, paths.samples()).stream()
                    .map(YTreeStringNode::getValue)
                    .filter(outdatedSampleIds::contains)
                    .collect(Collectors.toList());

            samplesToDelete.forEach(each -> deleteDirectory(ytTransaction, paths.sample(each)));

            LOG.info("Outdated samples deleted: {}", samplesToDelete.size());
            return samplesToDelete;
        }));
    }

    @Override
    public List<String> getGroups(String id) {
        return Caching.fetch(groupsCache, id, () -> {
            Sample sample = getSample(id);
            switch (sample.getType()) {
                case REGULAR:
                    return getGroupsRegular(sample);
                case AUDIENCE:
                    return getGroupsAudience();
                default:
            }
            return getGroupsRegular(sample);
        });
    }

    private List<String> getGroupsAudience() {
        return audience.getSegments();
    }

    private List<String> getGroupsRegular(Sample sample) {
        List<TSampleStats> results = new ArrayList<>();
        yt().tables().selectRows(
                YtQueries.selectSampleGroups(paths.sampleStats().toString(), sample.getId()),
                YTableEntryTypes.proto(TSampleStats.class),
                (Consumer<TSampleStats>) results::add
        );
        return results.stream().map(TSampleStats::getGroupID).collect(Collectors.toList());
    }

    @Override
    public List<Sample> getAll() {
        return tables()
                .samples()
                .selectAccessible()
                .fetchInto(Sample.class);
    }

    @Override
    public List<TSampleView> getViews(String id) {
        List<TSampleView> views = tables()
                .sampleViews()
                .selectBySampleIdQuery(id)
                .fetchInto(TSampleView.class);
        if (views.isEmpty()) {
            throw Exceptions.notFound();
        }
        return views;
    }

    private Optional<TSampleView> fetchView(String sampleId, String viewId) {
        return tables()
                .sampleViews()
                .selectByIdQuery(sampleId, viewId)
                .fetchOptionalInto(TSampleView.class);
    }

    @Override
    public TSampleView getView(String sampleId, String viewId) {
        return fetchView(sampleId, viewId).orElseThrow(Exceptions::notFound);
    }

    @Override
    public long getViewMinimalSize(String sampleId, String viewId) {
        TSampleView view = getView(sampleId, viewId);
        return Acl.getMinimalSize(view.getOptions().getMatching());
    }

    @Override
    public TSampleView deleteView(String sampleId, String viewId) {
        throw Exceptions.illegal("Not implemented");
    }

    @Override
    public TSampleView updateViewState(String sampleId, String viewId, ESampleViewState state) {
        return withSqlTransaction(tables -> {
            tables.sampleViews().updateStateQuery(sampleId, viewId, state).execute();
            return fetchView(tables, sampleId, viewId);
        });
    }

    @Override
    public TSampleView updateViewError(String sampleId, String viewId, String error) {
        return withSqlTransaction(tables -> {
            tables.sampleViews().updateErrorQuery(sampleId, viewId, error).execute();
            return fetchView(tables, sampleId, viewId);
        });
    }

    @Override
    public List<JobView> getSampleJobs(String id) {
        QuartzScheduler quartz = schedulers.getQuartz();
        return quartz.getJobs(id).stream().map(JobView::new).collect(Collectors.toList());
    }

    private Sample fetchSample(Tables tables, String id) {
        return tables
                .samples()
                .selectByIdQuery(id)
                .fetchOptionalInto(Sample.class)
                .orElseThrow(Exceptions::notFound);
    }

    @Override
    public DefaultSampleService clone() {
        return new DefaultSampleService(environment(), sql(), ytService(), schedulers, audience, siberiaClient);
    }

    @Override
    public List<JsonNode> getSamplePagesFromYt(String sampleId, String viewId, long page, int pageSize) {
        YPath table = YPath.simple(getView(sampleId, viewId).getPath());

        long offset = (page - 1) * pageSize;
        YPath ypathWithRange = table.withRange(offset, offset + pageSize);
        List<JsonNode> records = new ArrayList<>();
        yt().tables().read(ypathWithRange, YTableEntryTypes.JACKSON, (Consumer<JsonNode>) records::add);

        return records;
    }

    @Override
    public RawResponse getSampleFromSiberia(String userSetId, String lastUserId, long limit) {
        return siberiaClient.tryCall(siberia -> siberia.getUsers(userSetId, limit, lastUserId));
    }

    @Override
    public RawResponse describeBySiberia(String userSetId) {
        return siberiaClient.tryCall(siberia -> siberia.describeUserSet(userSetId));
    }

    @Override
    public TDescribeIdsResponse describeIdsBySiberia(TIds ids) {
        var response = siberiaClient.tryCall(siberia -> {
            var body = RequestBody.create(APPLICATION_JSON, JsonFormat.printer().print(ids));
            DescribingExperiment.TDescribingExperiment experiment = DescribingExperiment.TDescribingExperiment.newBuilder().setCryptaIdUserDataVersion("by_crypta_id").build();
            return siberia.describeIds(body, "fast", JsonFormat.printer().print(experiment));
        });

        var responseCode = response.getHttpCode();
        if (responseCode != 200) {
            throw Exceptions.internal(
                    "user_sets/describe_ids Siberia handle return code = '" + responseCode
                            + "' and message = '" + response.getText() + "'");
        }

        try {
            var builder = TDescribeIdsResponse.newBuilder();
            JsonFormat.parser().merge(response.getText(), builder);
            return builder.build();
        } catch (IOException e) {
            throw Exceptions.internal("Error while parsing response from Siberia: " + e.getMessage());
        }
    }

    @Override
    public Optional<TStats> getStatsFromSiberia(String userSetId) {
        return getStatsFromSiberia(userSetId, getLogin());
    }

    @Override
    public Optional<TStats> getStatsFromSiberia(String userSetId, String login) {
        var response = siberiaClient.tryCall(siberia -> siberia.getUserSetStats(userSetId));
        var responseCode = response.getHttpCode();
        if (responseCode == 404) {
            return Optional.empty();
        } else if (responseCode != 200) {
            throw Exceptions.internal(
                    "user_sets/get_stats Siberia handle return code = '" + responseCode + "' and message = '"
                            + response.getText() + "'");
        }

        try {
            var builder = TStats.newBuilder();
            JsonFormat.parser().merge(response.getText(), builder);
            return Optional.of(builder.build());
        } catch (IOException e) {
            throw Exceptions.internal("Error while parsing response from Siberia: " + e.getMessage());
        }
    }

    @Override
    public RawResponse createSiberiaSegment(String userSetId, String title, String rule) {
        return siberiaClient.tryCall(siberia -> siberia.createSegment(userSetId, title, rule));
    }

    @Override
    public RawResponse removeSiberiaSegment(String userSetId, String segmentId) {
        return siberiaClient.tryCall(siberia -> {
            var body = RequestBody.create(APPLICATION_JSON, segmentId);
            return siberia.deleteSegment(userSetId, body);
        });
    }

    @Override
    public RawResponse getSegmentsFromSiberia(String userSetId, String lastSegmentId, long limit) {
        return siberiaClient.tryCall(siberia -> siberia.getSegments(userSetId, limit, lastSegmentId));
    }

    @Override
    public RawResponse getSiberiaSegmentUsers(String userSetId, String segmentId, String lastUserId, long limit) {
        return siberiaClient
                .tryCall(siberia -> siberia.getSegmentUsers(userSetId, segmentId, limit, lastUserId));
    }

    @Override
    public RawResponse removeSiberiaUserSet(String userSetId) {
        return siberiaClient.tryCall(siberia -> siberia.deleteUserSet(userSetId));
    }

    private String getLogin() {
        return securityContext().getUserPrincipal().getName();
    }

    @Override
    public Subsamples getSubsamples(String sampleId) {
        return tables().subsamples()
                .selectBySampleIdQuery(sampleId)
                .fetchOptionalInto(Subsamples.class)
                .orElseThrow(Exceptions::notFound);
    }

    @Override
    public Subsamples createSubsamples(String sampleId, Map<String, String> userSetIdToGroupingKeyValue) {
        return withSqlTransaction(tables -> {
            try {
                tables().subsamples()
                        .insertQuery(
                                Subsamples.newBuilder()
                                        .setSampleId(sampleId)
                                        .putAllUserSetIdToGroupingKeyValue(userSetIdToGroupingKeyValue)
                                        .build()
                        ).execute();
            } catch (DataAccessException e) {
                throw Exceptions.wrongRequestException(
                        "Probably there's no such sample id in the parent table",
                        "DATABASE_ERROR"
                );
            }

            return getSubsamples(sampleId);
        });
    }

    @Override
    public String getUserSetBySegmentGroupId(String sampleId, Optional<String> groupId) {
        Sample sample = getSample(sampleId);
        if (groupId.isPresent()) {
            Map<String, String> userSetIdToGroup = sample.getUserSetIdToGroupingKeyValueMap();
            Map<String, String> groupToUserSetId = new HashMap<>();
            for (final Entry<String, String> e : userSetIdToGroup.entrySet()) {
                groupToUserSetId.put(e.getValue(), e.getKey());
            }
            return groupToUserSetId.get(groupId.get());
        } else {
            return sample.getSiberiaUserSetId();
        }
    }
}
