package ru.yandex.crypta.lab.yt;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.audience.AudienceClient;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.common.ws.EntityId;
import ru.yandex.crypta.lab.I18Utils;
import ru.yandex.crypta.lab.SegmentExportService;
import ru.yandex.crypta.lab.base.BaseService;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.tables.Tables;
import ru.yandex.crypta.lib.proto.EEnvironment;

public class DefaultSegmentExportService extends BaseService<SegmentExportService> implements SegmentExportService {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultLabService.class);
    private static final String AUDIENCE_LOGIN = "ya.crypta";
    private static final List<Long> INTERESTS_KEYWORDS = Arrays.asList(601L, 602L);
    private static final List<Long> SEGMENTS_KEYWORDS = Arrays.asList(544L, 545L, 546L, 547L, 548L, 549L);
    private static final List<Long> OLD_SEGMENTS_KEYWORDS = Arrays.asList(216L, 217L);
    private static final List<List<Long>> KEYWORDS = Arrays.asList(
            INTERESTS_KEYWORDS,
            SEGMENTS_KEYWORDS,
            OLD_SEGMENTS_KEYWORDS
    );

    private final AudienceClient audience;

    @Inject
    public DefaultSegmentExportService(EEnvironment environment, PostgresClient sql, AudienceClient audience)
    {
        super(environment, sql);
        this.audience = audience;
    }

    private Optional<Segment.Export> fetchExport(Tables tables, String id) {
        return tables
                .segmentExports()
                .selectQuery(id)
                .fetchOptionalInto(Segment.Export.class);
    }

    private Optional<Segment.Tags> fetchTags(Tables tables, String id) {
        return tables
                .segmentExportsTags()
                .selectQuery(id)
                .fetchOptionalInto(Segment.Tags.class);
    }

    private Optional<Segment.Export> fetchModifiableExport(Tables tables, String id) {
        return Optional.ofNullable(tables
                .segmentExports()
                .selectModifiableQuery(id)
                .fetchInto(Segment.Export.class)
                .get(0)
        );
    }

    private Segment fetchModifiableSegment(Tables tables, String id) {
        return tables
                .segments()
                .selectByIdModifiableQuery(id)
                .fetchOptionalInto(Segment.class)
                .orElseThrow(NotFoundException::new);
    }

    @Override
    public Segment.Export createExport(String segmentId, boolean exportCryptaId, Segment.Export.Builder export)
    {
        return withSqlTransaction(tables -> {
            Segment segment = fetchModifiableSegment(tables, segmentId);

            I18Utils.I18String nameUnpacked = I18Utils.unpack(segment.getName());
            String name = language().isRu() ? nameUnpacked.getRu() : nameUnpacked.getEn();

            String id = new EntityId("export").toString();
            export.setId(id);

            if (export.getSegmentId() == 0L && export.getKeywordId() > 0L && export.getKeywordId() != 557L) {
                export.setSegmentId(getAvailableSegmentId(export.getKeywordId()));
            }

            if (export.getSegmentId() == 0L && export.getKeywordId() == 557L) {
                JsonNode uploadingSegment = audience.uploadSegmentFile(AUDIENCE_LOGIN, exportCryptaId);

                JsonNode audienceSegmentId = uploadingSegment.get("segment").get("id");
                export.setSegmentId(audienceSegmentId.asLong());
                audience.confirmSegment(audienceSegmentId.asInt(), name, AUDIENCE_LOGIN, exportCryptaId);
            }

            if (exportCryptaId) {
                export.setExportTypeId("crypta_id");
            } else {
                export.setExportTypeId("yandexuid");
            }

            tables.segmentExports().insertQuery(segmentId, export).execute();

            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export getExport(String id) {
        return fetchExport(tables(), id).orElseThrow(Exceptions::notFound);
    }

    @Override
    public Segment.Export bindExport(String id, String segmentId) {
        return withSqlTransaction(tables -> {
            Segment.Export existingExport = fetchModifiableExport(tables, id).orElseThrow(Exceptions::notFound);
            Segment existingSegment = fetchModifiableSegment(tables, segmentId);
            tables.segmentExports().updateSegmentIdQuery(existingExport.getId(), existingSegment.getId()).execute();
            return fetchModifiableExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export updateExport(String id, Segment.Export.Builder export) {
        return withSqlTransaction(tables -> {
            Segment.Export existing = fetchModifiableExport(tables, id).orElseThrow(Exceptions::notFound);
            Segment.Export.Builder merged = existing.toBuilder().mergeFrom(export.build());
            tables.segmentExports().updateQuery(id, merged).execute();
            return fetchModifiableExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export deleteExport(String id) {
        return withSqlTransaction(tables -> {
            var dependantExports = getDependantExports(id);
            if (!dependantExports.isEmpty()) {
                var dependantExportIds = dependantExports.stream()
                        .map(Segment.Export::getId)
                        .collect(Collectors.joining(", "));
                throw Exceptions.wrongRequestException(
                        String.format("Экспорт %s используется в выражениях следующих экспортов: %s. " +
                                        "Пожалуйста, удалите экспорт из выражений и попробуйте удалить еще раз.",
                                id,
                                dependantExportIds),
                        "405"
                );
            }

            Segment.Export export = fetchExport(tables, id).orElseThrow(Exceptions::notFound);
            LOG.info("Deleting export {}", export);
            updateExportState(id, Segment.Export.State.DELETED);

            if (export.getKeywordId() == 557L) {
                audience.deleteSegment(String.valueOf(export.getSegmentId()));
            }
            return export;
        });
    }

    @Override
    public Segment.Export updateExportState(String id, Segment.Export.State state) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().updateExportStateQuery(id, state).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export updateExportNextActivityCheckTs(String id, Long timestamp) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().updateExportNextActivityCheckTs(id, timestamp).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export updateExportBigbCoverage(String id, Long value, Long timestamp) {
        Segment.Export.Builder export = getExport(id).toBuilder();
        export.getCoveragesBuilder()
                .getBigbBuilder()
                .setValue(value)
                .setTimestamp(timestamp);
        if (export.getState() == Segment.Export.State.CREATED && value > 0) {
            export.setState(Segment.Export.State.ACTIVE);
        }
        return updateExport(id, export);
    }

    @Override
    public Segment.Export updateExportProfilesCoverage(String id, Long value, Long timestamp) {
        Segment.Export.Builder export = getExport(id).toBuilder();
        export.getCoveragesBuilder()
                .getProfilesBuilder()
                .setValue(value)
                .setTimestamp(timestamp);
        if (export.getState() == Segment.Export.State.CREATED && value > 0) {
            export.setState(Segment.Export.State.ACTIVE);
        }
        return updateExport(id, export);
    }

    @Override
    public Segment.Export putExportExpression(String id, List<String> expressions) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().updateExpressionQuery(id, expressions).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export deleteExportExpression(String id) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().deleteExpressionQuery(id).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export putLal(String id, String lal) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().updateLalQuery(id, lal).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export deleteLal(String id) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().deleteLalQuery(id).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export putExportRuleId(String id, String ruleId) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().updateRuleIdQuery(id, ruleId).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export deleteExportRuleId(String id) {
        return withSqlTransaction(tables -> {
            tables.segmentExports().deleteRuleIdQuery(id).execute();
            return fetchExport(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public List<Segment.Export> getExportsWithRuleId() {
        return tables()
                .segmentExports()
                .selectQueryWithRuleId()
                .fetchInto(Segment.Export.class);
    }

    @Override
    public List<Segment.Export> getExportsNotExportedToBigB() {
        return tables()
                .segmentExports()
                .selectNotExportedToBigBQuery()
                .fetchInto(Segment.Export.class);
    }

    @Override
    public Segment.Export disableExportToBigB(String exportId) {
        return withSqlTransaction(tables -> {
            tables().segmentExports().disableExportToBigbQuery(exportId).execute();
            return fetchExport(tables, exportId).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Export enableExportToBigB(String exportId) {
        return withSqlTransaction(tables -> {
            tables().segmentExports().enableExportToBigbQuery(exportId).execute();
            return fetchExport(tables, exportId).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Long getAvailableSegmentId(Long keywordId) {
        List<Long> keywordIds = getKeywordIdGroup(keywordId);
        return Optional.ofNullable(
                tables().segmentExports()
                        .selectLastSegmentIdQuery(keywordIds)
                        .fetchOne(0, Long.class)
        ).orElse(0L) + 1;
    }

    private List<Long> getKeywordIdGroup(Long keywordId) {
        return KEYWORDS.stream()
                .filter(item -> item.contains(keywordId))
                .findFirst()
                .orElse(Arrays.asList(keywordId));
    }

    @Override
    public List<Segment.Export> getDependantExports(String exportId) {
        return tables()
                .segmentExports()
                .selectQueryWithExportIdInExpressions(exportId)
                .fetchInto(Segment.Export.class);
    }

    @Override
    public DefaultSegmentExportService clone() {
        return new DefaultSegmentExportService(environment(), sql(), audience);
    }


    @Override
    public Segment.Tags getTags(String id) {
        return withSqlTransaction(tables -> {
            return fetchTags(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Tags addTag(String id, String tag) {
        return withSqlTransaction(tables -> {
            tables().segmentExportsTags().insertQuery(id, tag).execute();
            return fetchTags(tables, id).orElseThrow(Exceptions::notFound);
        });
    }

    @Override
    public Segment.Tags removeTag(String id, String tag) {
        tables().segmentExportsTags().deleteQuery(id, tag).execute();
        return getTags(id);
    }
}
