package ru.yandex.crypta.lab.yt;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.ws.rs.core.SecurityContext;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.google.common.base.Splitter;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Iterables;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import javafx.util.Pair;
import org.jooq.Record;
import org.jooq.RecordMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.clients.audience.AudienceClient;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.common.ws.EntityId;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.I18Utils;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.SegmentService;
import ru.yandex.crypta.lab.base.BaseService;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.SegmentAttributes;
import ru.yandex.crypta.lab.proto.SegmentConditions;
import ru.yandex.crypta.lab.proto.SegmentGroup;
import ru.yandex.crypta.lab.proto.SegmentIdPair;
import ru.yandex.crypta.lab.proto.TSimpleSampleStatsWithInfo;
import ru.yandex.crypta.lab.proto.Timestamps;
import ru.yandex.crypta.lab.proto.Translation;
import ru.yandex.crypta.lab.proto.Translations;
import ru.yandex.crypta.lab.siberia.SiberiaClient;
import ru.yandex.crypta.lab.tables.ResponsiblesTable;
import ru.yandex.crypta.lab.tables.SegmentExportsTable;
import ru.yandex.crypta.lab.tables.SegmentIdsTable;
import ru.yandex.crypta.lab.tables.SegmentsTable;
import ru.yandex.crypta.lab.tables.StakeholdersTable;
import ru.yandex.crypta.lab.tables.Tables;
import ru.yandex.crypta.lab.utils.ModelSegmentRelations;
import ru.yandex.crypta.lab.utils.SegmentName;
import ru.yandex.crypta.lab.utils.SegmentNode;
import ru.yandex.crypta.lab.utils.Tanker;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.proto.TTankerConfig;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

public class DefaultSegmentService extends BaseService<SegmentService> implements SegmentService {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultLabService.class);
    private final TTankerConfig tankerConfig;
    private final SiberiaClient siberiaClient;
    private final YtService yt;
    private final LabService lab;
    private final AudienceClient audience;

    private static final Cache<Integer, Map<String, String>> segmentToUserSetCache = CacheBuilder.newBuilder()
            .expireAfterWrite(1, TimeUnit.HOURS)
            .build();

    @Inject
    public DefaultSegmentService(
            EEnvironment environment,
            PostgresClient sql,
            TTankerConfig tankerConfig,
            SiberiaClient siberiaClient,
            YtService yt,
            LabService lab,
            AudienceClient audience
    )
    {
        super(environment, sql);
        this.tankerConfig = tankerConfig;
        this.siberiaClient = siberiaClient;
        this.yt = yt;
        this.lab = lab;
        this.audience = audience;
    }

    private String localize(String packed) {
        I18Utils.I18String unpacked = I18Utils.unpack(packed);
        if (language().isEn()) {
            return unpacked.getEn();
        } else if (language().isRu()) {
            return unpacked.getRu();
        } else {
            throw Exceptions.illegal("Unknown language");
        }
    }

    private Segment localizeSegment(Segment.Builder builder) {
        builder.setName(localize(builder.getName()));
        builder.setDescription(localize(builder.getDescription()));

        return builder.build();
    }

    private Segment readSegmentLocalized(Record record) {
        return localizeSegment(tables().segments().readSegment(record).toBuilder());
    }

    private SegmentAttributes readSegmentAttributesLocalized(Record record) {
        SegmentAttributes.Builder attributesBuilder = SegmentsTable.readSegmentAttributes(record).toBuilder();
        attributesBuilder.setName(localize(attributesBuilder.getName()));

        return attributesBuilder.build();
    }

    private SegmentGroup localizeGroup(SegmentGroup.Builder builder) {
        builder.setName(localize(builder.getName()));
        builder.setDescription(localize(builder.getDescription()));
        return builder.build();
    }

    private SegmentGroup readGroupLocalized(Record record) {
        return localizeGroup(SegmentsTable.readGroup(record).toBuilder());
    }

    private Tanker getTanker() {
        return new Tanker(environment(), tankerConfig.getToken());
    }

    @Override
    public boolean allowedToEditSegment(String id, Provider<SecurityContext> context) {
        var segment = get(id);
        var userName = context.get().getUserPrincipal().getName();

        return context.get().isUserInRole(Roles.Lab.EXTENDED) ||
                context.get().isUserInRole(Roles.Lab.ADMIN) ||
                userName.equals(segment.getAuthor()) ||
                segment.getResponsiblesList().stream().map(each -> Iterables.get(Splitter.on('@').split(each), 0).toLowerCase()).toList().contains(userName);
    }

    @Override
    public List<Segment> getAll() {
        return tables()
                .segments()
                .selectQuery()
                .fetch(this::readSegmentLocalized);
    }

    @Override
    public List<SegmentAttributes> getSegmentsAttributes() {
        return tables()
                .segments()
                .selectAttributesQuery()
                .fetch(this::readSegmentAttributesLocalized);
    }

    @Override
    public SegmentAttributes getByExportId(String exportId) {
        return tables()
                .segments()
                .selectByExportIdQuery(exportId)
                .fetchOptional(this::readSegmentAttributesLocalized)
                .orElseThrow(Exceptions::notFound);
    }

    @Override
    public List<SegmentAttributes> matchSearchItems(List<String> searchItems) {
        return tables().segments().selectByIdsFuzzyQuery(searchItems).fetch(this::readSegmentAttributesLocalized);
    }

    @Override
    public List<SegmentAttributes> matchSegmentConditions(SegmentConditions conditions) {
        return tables()
                .segments()
                .selectAttributesWithConditionsQuery(conditions)
                .fetch(this::readSegmentAttributesLocalized);
    }

    @Override
    public List<SegmentGroup> getAllGroups() {
        return tables()
                .segments()
                .selectGroupQuery()
                .fetch(this::readGroupLocalized);
    }

    private List<SegmentGroup> getAllGroupsOnly() {
        return tables()
                .segments()
                .selectGroupOnlyQuery()
                .fetch(this::readGroupLocalized);
    }

    private SegmentGroup getRootSegmentGroup() {
        return tables()
                .segments()
                .selectGroupByIdQuery("root")
                .fetchOne(this::readGroupLocalized);
    }

    private void checkBlankElement(List<String> list, String elementName) {
        if (list.stream().anyMatch(String::isBlank)) {
            throw Exceptions.wrongRequestException(String.format("%s must not be blank", elementName), "VALIDATION_FAILED");
        }
    }

    @Override
    public Segment createSegment(String preferredId, Segment.Builder prototype) {
        checkBlankElement(prototype.getTicketsList(), "Ticket");
        checkBlankElement(prototype.getResponsiblesList(), "Responsible");
        checkBlankElement(prototype.getStakeholdersList(), "Stakeholder");

        try {
            getAllTypes(prototype.getParentId());
        } catch (NotFoundException e) {
            throw Exceptions.wrongRequestException(String.format("Parent segment with id %s is not found", prototype.getParentId()), "NOT_FOUND");
        }

        return withSqlTransaction(tables -> {
            String id = Optional
                    .ofNullable(preferredId)
                    .map(x -> "segment-" + x)
                    .orElseGet(() -> new EntityId("segment").toString());

            long timestamp = Instant.now().getEpochSecond();

            Segment segment = prototype
                    .setId(id)
                    .setTimestamps(Timestamps.newBuilder().setCreated(timestamp).setModified(timestamp))
                    .setAuthor(securityContext().getUserPrincipal().getName())
                    .build();
            LOG.info("Creating segment {}", segment);

            tables.segments().insertQuery(segment).execute();
            storeSegmentReferences(segment, tables);
            return fetchSegment(segment.getId(), tables);
        });
    }

    @Override
    public Segment createUserSegment(Segment.Builder prototype) {
        prototype.setScope(Segment.Scope.INTERNAL)
                .setType(Segment.Type.USER_SEGMENT)
                .setParentId("root-users")
                .setAuthor(securityContext().getUserPrincipal().getName())
                .addResponsibles(securityContext().getUserPrincipal().getName() + "@yandex-team.ru");
        return createSegment(null, prototype);
    }

    private List<String> getCorrespondingAudienceSegmentsIds(Segment.Builder segment) {
        var exports = segment.getExports().getExportsList();
        return exports.stream()
                .map(export -> {
                    if (export.getKeywordId() == 557L) {
                        return String.valueOf(export.getSegmentId());
                    } else {
                        return null;
                    }
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private void checkExports(Segment.Builder segment) {
        var exports = segment.getExports().getExportsList();
        exports.forEach(export -> {
            var dependantExports = lab.segmentExports().getDependantExports(export.getId());
            dependantExports.forEach(dependantExport -> {
                var dependantExportSegmentId = getByExportId(dependantExport.getId()).getId();

                // Check if the dependantExport in the same segment. If not, throw an exception.
                if (!dependantExportSegmentId.equals(segment.getId())) {
                    throw Exceptions.wrongRequestException(
                            String.format("Сегмент %s, который вы пытаетесь удалить, содержит экспорт %s, использумый в выражениях следующих экспортов: %s. " +
                                    "Пожалуйста, удалите экспорт из выражений и попробуйте удалить еще раз.",
                                    segment.getId(),
                                    export.getId(),
                                    dependantExports.stream()
                                            .map(Segment.Export::getId)
                                            .collect(Collectors.joining(", "))),
                            "405"
                    );
                }
            });
        });
    }

    @Override
    public Segment deleteSegment(String id) {
        return withSegmentForUpdate(id, (tables, segment) -> {
            LOG.info("Deleting segment {}", segment);

            checkExports(segment);
            var audienceSegmentsToDelete = getCorrespondingAudienceSegmentsIds(segment);

            segment.getExports().getExportsList().forEach(export -> {
                tables.segmentExports().updateExportStateQuery(export.getId(), Segment.Export.State.DELETED).execute();
            });

            segment.setState(Segment.State.DELETED);
            tables.segments().updateQuery(segment.build()).execute();
            audienceSegmentsToDelete.forEach(audience::deleteSegment);

            var localizedSegment = localizeSegment(segment);
            return localizedSegment;
        });
    }

    @Override
    public SegmentNode getGroupsTree() {
        Queue<SegmentNode> segmentNodes = new ArrayDeque<>();
        getAllGroups()
                .forEach(group -> segmentNodes.add(
                        new SegmentNode(
                                group.getId(),
                                group.getParentId(),
                                new ArrayList<>(),
                                group.getName())
                        )
                );

        SegmentGroup rootSegmentGroup = getRootSegmentGroup();
        SegmentNode rootSegmentNode = new SegmentNode(
                rootSegmentGroup.getId(),
                rootSegmentGroup.getParentId(),
                new ArrayList<>(),
                rootSegmentGroup.getName());

        // Remove root segment node to avoid root duplicating in a tree
        while (!segmentNodes.isEmpty()) {
            SegmentNode offerRoot = segmentNodes.remove();
            if (offerRoot.getId().equals("root")) {
                break;
            } else {
                segmentNodes.add(offerRoot);
            }
        }

        // You must have ready root node to avoid eternal loop
        while (!segmentNodes.isEmpty()) {
            SegmentNode node = segmentNodes.remove();
            SegmentNode newNode = rootSegmentNode.insertNode(node);
            if (newNode.getId().equals(node.getId())) {
                segmentNodes.add(newNode);
            }
        }

        return rootSegmentNode;
    }

    @Override
    public SegmentGroup getGroup(String groupId) throws NotFoundException {
        return fetchSegmentGroup(groupId, tables(), true);
    }

    @Override
    public SegmentGroup getGroup(String groupId, boolean localized) {
        return fetchSegmentGroup(groupId, tables(), localized);
    }

    @Override
    public SegmentGroup deleteGroup(String groupId) {
        return withSqlTransaction(tables -> {
            SegmentGroup existing = fetchSegmentGroupForUpdate(groupId, tables);
            tables.segments().deleteGroupByIdQuery(groupId).execute();
            return localizeGroup(existing.toBuilder());
        });
    }

    @Override
    public SegmentGroup createGroup(SegmentGroup.Builder prototype) {
        List<String> groupIds = getAllGroups()
                .stream()
                .map(SegmentGroup::getId)
                .collect(Collectors.toList());

        if (!groupIds.contains(prototype.getParentId())) {
            throw Exceptions.illegal("Unknown parent groupId, segment group not created");
        }
        return withSqlTransaction(tables -> {
            EntityId id = new EntityId("group");
            String parentId = prototype.getParentId();

            SegmentGroup segmentGroup = prototype
                    .setId(id.toString())
                    .setParentId(parentId)
                    .setAuthor(securityContext().getUserPrincipal().getName())
                    .build();
            LOG.info("Creating segment group {}", id);

            tables.segments().insertGroupQuery(segmentGroup).execute();
            return fetchSegmentGroup(segmentGroup.getId(), tables, true);
        });
    }

    @Override
    public Segment setSegmentParent(String segmentId, String newParentId) {
        if (Objects.equals(segmentId, newParentId)) {
            throw Exceptions.illegal("Segment might not have itself as a parent");
        }
        return withSegmentForUpdate(segmentId, (tables, segment) -> {
            tables.segments().updateParentIdQuery(segmentId, newParentId).execute();
            return localizeSegment(segment);
        });
    }

    @Override
    public SegmentGroup setGroupParent(String groupId, String newParentId) {
        return withSqlTransaction(tables -> {
            SegmentGroup segmentGroup = fetchSegmentGroupForUpdate(groupId, tables);
            tables.segments().updateParentIdQuery(groupId, newParentId).execute();
            return localizeGroup(segmentGroup.toBuilder());
        });
    }

    @Override
    public Segment get(String id) throws NotFoundException {
        return fetchSegment(id, tables());
    }

    public Segment getAllTypes(String id) throws NotFoundException {
        return fetchSegmentAllTypes(id, tables());
    }

    @Override
    public Segment getNotLocalized(String id) throws NotFoundException {
        return fetchRawSegment(id, tables());
    }

    @Override
    public Segment updateName(String id, String value) {
        LOG.debug("updateName {} to {}", id, value);
        checkBlankRuI18(value, "Name");
        return updateSegment(id, segment -> segment.setName(value));
    }

    @Override
    public Segment updateDescription(String id, String value) {
        checkBlankRuI18(value, "Description");
        return updateSegment(id, segment -> segment.setDescription(value));
    }

    @Override
    public Segment updateNameAndDescription(String id, String name, String description) {
        checkBlankRuI18(name, "Name");
        checkBlankRuI18(description, "Description");
        return updateSegment(id, segment -> segment.setName(name).setDescription(description));
    }

    private void checkBlankRuI18(String packed, String name) {
        I18Utils.I18String i18String = I18Utils.unpack(packed);
        if (i18String.getRu().isBlank()) {
            throw Exceptions.wrongRequestException(String.format("%s must not be blank", name), "VALIDATION_FAILED");
        }
    }

    @Override
    public Segment updateNameKey(String id, String key) {
        return updateSegment(id, segment -> segment.setTankerNameKey(key));
    }

    @Override
    public Segment updateDescriptionKey(String id, String key) {
        return updateSegment(id, segment -> segment.setTankerDescriptionKey(key));
    }

    @Override
    public Map<String, List<SegmentGroup>> getParentsPerSegment() {
        List<Segment> segments = getAll();
        Map<String, List<SegmentGroup>> groups = getAllGroups()
                .stream()
                .collect(
                        Collectors.groupingBy(SegmentGroup::getId)
                );

        Map<String, List<SegmentGroup>> result = new HashMap<>();

        segments.forEach(segment -> {
            String segmentId = segment.getId();
            List<SegmentGroup> parents = result.getOrDefault(segmentId, new ArrayList<>());
            Queue<String> queue = new ArrayDeque<>();
            queue.add(segment.getParentId());
            while (!queue.isEmpty()) {
                String parentId = queue.poll();
                parents.add(groups.get(parentId).get(0));
                List<SegmentGroup> segmentGroups = groups.get(parentId);
                String parentIdOfParent = segmentGroups.get(0).getParentId();
                if (!parentIdOfParent.equals(parentId)) {
                    queue.add(parentIdOfParent);
                }
            }
            result.put(segmentId, parents);
        });

        return result;
    }

    @Override
    public Map<Long, Map<Long, Segment>> getExportsToSimpleSegments() {
        Map<Long, Map<Long, Segment>> mapping = new HashMap<>();
        tables()
                .segments()
                .selectExportsToSimpleSegments()
                .fetch()
                .forEach(record -> {
                    String id = record.get(SegmentExportsTable.ID);
                    long keywordId = record.get(SegmentExportsTable.EXPORT_KEYWORD_ID);

                    Long segmentId = record.get(SegmentExportsTable.EXPORT_SEGMENT_ID);
                    if (segmentId == null) {
                        segmentId = 0L;
                    }

                    Segment.Export.Type exportType = Segment.Export.Type.valueOf(record.get(SegmentExportsTable.TYPE));

                    mapping.putIfAbsent(keywordId, new HashMap<>());
                    String nameEn = record.get(SegmentsTable.NAME_EN);
                    String nameRu = record.get(SegmentsTable.NAME_RU);
                    String descriptionEn = record.get(SegmentsTable.DESCRIPTION_EN);
                    String descriptionRu = record.get(SegmentsTable.DESCRIPTION_RU);
                    Segment.Builder segment = Segment.newBuilder()
                            .setId(record.get(SegmentsTable.ID))
                            .setName(localize(I18Utils.pack(I18Utils.string(nameEn, nameRu))))
                            .setDescription(localize(I18Utils.pack(I18Utils.string(descriptionEn, descriptionRu))))
                            .setScope(Segment.Scope.valueOf(record.get(SegmentsTable.SCOPE)))
                            .setType(Segment.Type.valueOf(record.get(SegmentsTable.TYPE)));
                    segment.getExportsBuilder()
                            .addExportsBuilder()
                            .setId(id)
                            .setKeywordId(keywordId)
                            .setSegmentId(segmentId)
                            .setType(exportType);
                    mapping.get(keywordId).put(segmentId, segment.build());
                });
        return mapping;
    }

    @Override
    public void updatePriority(String id, Long priority) {
        tables().segments().updatePriorityQuery(id, priority).execute();
    }

    @Override
    public void addStakeholder(String segmentId, String stakeholder) {
        withSqlTransaction(tables -> {
            if (tables().stakeholders().selectBySegmentIdAndStakeholderQuery(segmentId, stakeholder).fetch().size()
                    == 0)
            {
                long createdTimestamp = Instant.now().getEpochSecond();
                tables().stakeholders().insertQuery(stakeholder, segmentId, createdTimestamp, createdTimestamp)
                        .execute();
            }

            return null;
        });
    }

    @Override
    public void deleteStakeholder(String segmentId, String stakeholder) {
        withSqlTransaction(tables -> {
            if (tables().stakeholders()
                    .selectBySegmentIdAndStakeholderQuery(segmentId, stakeholder).fetch().size() > 0)
            {
                tables.stakeholders().deleteStakeholderQuery(segmentId, stakeholder).execute();
            }

            return null;
        });
    }

    @Override
    public void addResponsible(String segmentId, String responsible) {
        withSqlTransaction(tables -> {
            if (tables().responsibles().selectBySegmentIdAndResponsibleQuery(segmentId, responsible).fetch().size()
                    == 0)
            {
                long createdTimestamp = Instant.now().getEpochSecond();
                tables().responsibles().insertQuery(responsible, segmentId, createdTimestamp, createdTimestamp)
                        .execute();
            }

            return null;
        });
    }

    @Override
    public void deleteResponsible(String segmentId, String responsible) {
        withSqlTransaction(tables -> {
            if (tables().responsibles()
                    .selectBySegmentIdAndResponsibleQuery(segmentId, responsible).fetch().size() > 0)
            {
                tables.responsibles().deleteBySegmentIdAndResponsibleQuery(segmentId, responsible).execute();
            }

            return null;
        });
    }

    private Map<String, String> fetchSegmentToUserSetMap() {
        return tables().segmentIds().selectQuery().fetchInto(SegmentIdPair.class)
                .stream()
                .collect(Collectors.toMap(
                        SegmentIdPair::getSegmentLabId,
                        SegmentIdPair::getUserSetId
                ));
    }

    private String getUserSetId(String segmentId) {
        String userSetId = Caching.fetch(segmentToUserSetCache, 0, this::fetchSegmentToUserSetMap)
                .getOrDefault(segmentId, null);

        if (userSetId != null) {
            return userSetId;
        } else {
            throw Exceptions.notFound();
        }
    }

    @Override
    public Optional<TSimpleSampleStatsWithInfo> getStats(String id) {
        String userSetId = getUserSetId(id);
        return lab.withLanguage(language()).getStatsFromSiberia(userSetId);

    }

    @Override
    public Optional<TSimpleSampleStatsWithInfo> getStats(
                                                 String exportId,
                                                 Optional<String> baseExportId,
                                                 Optional<String> baseSampleId,
                                                 Optional<String> baseGroupId
    ) {
        String userSetId = lab.getUserSetIdByExportId(exportId);
        if (baseExportId.isEmpty() && baseSampleId.isEmpty()) {
            return lab.withLanguage(language()).getStatsFromSiberia(userSetId);
        }
        String baseUserSetId;
        if (baseExportId.isPresent()) {
            baseUserSetId = lab.getUserSetIdByExportId(baseExportId.get());
        } else {
            baseUserSetId = lab.samples().getUserSetBySegmentGroupId(baseSampleId.get(), baseGroupId);
        }
        return lab.withLanguage(language()).getStatsFromSiberia(new Pair<>(userSetId, baseUserSetId));
    }


    private JsonObject extractKeys(JsonObject jsonObject) {
        JsonObject keys = jsonObject
                .getAsJsonObject("keysets")
                .getAsJsonObject("profile")
                .getAsJsonObject("keys");

        if (keys.entrySet().isEmpty()) {
            throw Exceptions.notFound();
        }

        return keys;
    }

    private JsonObject extractTranslationsFromEntry(JsonObject entry) {
        return entry.getAsJsonObject("translations");
    }

    private JsonObject extractTranslations(JsonObject jsonObject, String key) {
        return extractTranslationsFromEntry(extractKeys(jsonObject).getAsJsonObject(key));
    }

    @Override
    public Translations getTranslationsFromTankerByKey(String tankerKey) {
        String tankerResponse = getTanker().getTranslationsByKey(tankerKey);
        JsonObject jsonTranslations = extractTranslations(
                JsonParser.parseString(tankerResponse).getAsJsonObject(), tankerKey
        );

        if (tankerResponse.isEmpty()) {
            throw Exceptions.notFound();
        }

        Translations.Builder translations = Translations.newBuilder().setTankerKey(tankerKey);

        JsonObject translationEn = jsonTranslations.getAsJsonObject("en");
        JsonObject translationRu = jsonTranslations.getAsJsonObject("ru");

        String textField = "form";
        String statusField = "status";
        String authorField = "author";

        translations.setEn(
                Translation.newBuilder()
                .setText(translationEn.get(textField).getAsString())
                .setStatus(translationEn.get(statusField).getAsString())
                .setAuthor(translationEn.get(authorField).getAsString())
        );

        translations.setRu(
                Translation.newBuilder()
                        .setText(translationRu.get(textField).getAsString())
                        .setStatus(translationRu.get(statusField).getAsString())
                        .setAuthor(translationRu.get(authorField).getAsString())
        );

        return translations.build();
    }

    private SegmentGroup updateGroupNameKey(String id, String key) {
        return updateGroup(id, group -> group.setTankerNameKey(key));
    }

    private SegmentGroup updateGroupDescriptionKey(String id, String key) {
        return updateGroup(id, group -> group.setTankerDescriptionKey(key));
    }

    @Override
    public JsonNode getTranslations() {
        String tankerResponse = getTanker().getKeysFromTanker();

        JsonNode translationsJson = JsonNodeFactory.instance.objectNode();

        if (tankerResponse.isEmpty()) {
            LOG.debug("Got empty response from Tanker");
            return translationsJson;
        }

        try {
            translationsJson = new ObjectMapper().readTree(tankerResponse);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }

        return translationsJson;
    }

    @Override
    public void updateKeysFromTanker() {
        LOG.debug("Updating segments keys from Tanker");

        Pattern segmentTankerKeyRegex = Pattern.compile("crypta_(\\S+)_(\\S+)");

        String tankerResponse = getTanker().getKeysFromTanker();
        if (tankerResponse.isEmpty()) {
            LOG.debug("Got empty response from Tanker");
            return;
        }

        Map<String, String> tankerNameKeys = new HashMap<>();
        Map<String, String> tankerDescriptionKeys = new HashMap<>();

        JsonObject keysObject = extractKeys(JsonParser.parseString(tankerResponse).getAsJsonObject());

        String language = "ru";

        for (Map.Entry<String, JsonElement> entry : keysObject.entrySet()) {
            String ruText = extractTranslationsFromEntry(entry.getValue().getAsJsonObject())
                    .getAsJsonObject(language)
                    .get("form")
                    .getAsString();

            if (!ruText.isEmpty()) {
                String tankerKey = entry.getKey();
                Matcher matcher = segmentTankerKeyRegex.matcher(tankerKey);
                if (matcher.matches()) {
                    String group1 = matcher.group(1);
                    String segmentId;

                    if (group1.startsWith("segment-") || group1.startsWith("group-")) {
                        segmentId = group1;
                    } else if (group1.startsWith("interest_")) {
                        segmentId = "segment-" + matcher.group(1).replaceFirst("^interest_", "");
                    } else {
                        LOG.info(String.format("Unknown key pattern: %s", group1));
                        segmentId = "segment-" + group1;
                    }

                    if (matcher.group(2).equals("name")) {
                        tankerNameKeys.putIfAbsent(segmentId, tankerKey);
                    } else if (matcher.group(2).equals("description")) {
                        tankerDescriptionKeys.putIfAbsent(segmentId, tankerKey);
                    }
                }
            }
        }

        List<String> segmentIds = tables().segments().selectAllQuery().fetch(this::readSegmentLocalized)
                .stream()
                .map(Segment::getId)
                .collect(Collectors.toList());

        List<String> groupIds = getAllGroupsOnly().stream().map(SegmentGroup::getId).collect(Collectors.toList());

        for (String segmentId : segmentIds) {
            if (tankerNameKeys.containsKey(segmentId)) {
                updateNameKey(segmentId, tankerNameKeys.get(segmentId));
            }

            if (tankerDescriptionKeys.containsKey(segmentId)) {
                updateDescriptionKey(segmentId, tankerDescriptionKeys.get(segmentId));
            }
        }

        for (String groupId : groupIds) {
            if (tankerNameKeys.containsKey(groupId)) {
                updateGroupNameKey(groupId, tankerNameKeys.get(groupId));
            }

            if (tankerDescriptionKeys.containsKey(groupId)) {
                updateGroupDescriptionKey(groupId, tankerDescriptionKeys.get(groupId));
            }
        }

        LOG.info("Segment keys updated from Tanker");
    }

    private String createTemporaryFile(String content) {
        LOG.debug("createTemporaryFile: {}", content);
        try {
            File f = File.createTempFile("tmp", ".txt");
            f.deleteOnExit();
            OutputStreamWriter writer = new OutputStreamWriter(
                    new FileOutputStream(f.getAbsolutePath()),
                    StandardCharsets.UTF_8
            );
            writer.write(content);
            writer.close();
            return f.getAbsolutePath();
        } catch (Exception e) {
            LOG.error("createTemporaryFile: {}", e.toString());
        }
        return null;
    }

    private String getTankerKey(String segmentId, Segment.Type type, String suffix) {
        String prefix = segmentId;

        if (Objects.equals(type, Segment.Type.INTEREST)) {
            prefix = segmentId.replace("segment-", "interest_");
        }
        return String.format("crypta_%s_%s", prefix, suffix);
    }

    private String getTankerNameKey(String segmentId, Segment.Type type) {
        return getTankerKey(segmentId, type, "name");
    }

    private String getTankerDescriptionKey(String segmentId, Segment.Type type) {
        return getTankerKey(segmentId, type, "description");
    }

    private void storeTranslationInTanker(String key, String text, boolean skipTranslation) {
        Tanker tanker = getTanker();
        tanker.sendPostRequestWithFile(
                createTemporaryFile(
                        tanker.createTankerCreateSegmentJson(key, text, skipTranslation)
                )
        );
    }

    @Override
    public Map<String, Translations> createSegmentKeysInTanker(String segmentId, String nameRu, String descriptionRu, boolean skipTranslation)
    {
        return withSqlTransaction(tables -> {
            Segment segment = getNotLocalized(segmentId);

            String tankerNameKey = getTankerNameKey(segmentId, segment.getType());
            String tankerDescriptionKey = getTankerDescriptionKey(segmentId, segment.getType());

            Segment segmentWithKeys = segment.toBuilder()
                    .setTankerNameKey(tankerNameKey)
                    .setTankerDescriptionKey(tankerDescriptionKey)
                    .build();

            tables.segments().updateQuery(segmentWithKeys).execute();

            storeTranslationInTanker(tankerNameKey, nameRu, skipTranslation);
            storeTranslationInTanker(tankerDescriptionKey, descriptionRu, skipTranslation);

            Map<String, Translations> translations = new HashMap<>();
            translations.put("name", getTranslationsFromTankerByKey(tankerNameKey));
            translations.put("description", getTranslationsFromTankerByKey(tankerDescriptionKey));

            return translations;
        });
    }

    @Override
    public Translations removeKeyAndTranslations(String key) {
        Tanker tanker = getTanker();
        Translations translationsToRemove = getTranslationsFromTankerByKey(key);

        tanker.deleteKeyFromTanker(key);
        return translationsToRemove;
    }

    @Override
    public JsonNode getTankerKeysInconsistency() {
        Map<String, HashSet<String>> result = new HashMap<>();
        HashSet<String> keysNotInLab = new HashSet<>();

        HashSet<String> segmentKeys = new HashSet<>();
        tables().segments().selectAllQuery().fetch(this::readSegmentLocalized).forEach(segment -> {
            String nameKey = segment.getTankerNameKey();
            String descriptionKey = segment.getTankerDescriptionKey();

            if (!Objects.equals(nameKey, null) && !nameKey.equals("")) {
                segmentKeys.add(segment.getTankerNameKey());
            }

            if (!Objects.equals(descriptionKey, null) && !descriptionKey.equals("")) {
                segmentKeys.add(segment.getTankerDescriptionKey());
            }
        });

        HashSet<String> tankerKeys = new HashSet<>();
        Tanker tanker = getTanker();
        String tankerResponse = tanker.getKeysFromTanker();

        if (!tankerResponse.isEmpty()) {
            JsonObject keysObject = extractKeys(JsonParser.parseString(tankerResponse).getAsJsonObject());
            tankerKeys = keysObject.entrySet().stream()
                    .map(Map.Entry::getKey)
                    .collect(Collectors.toCollection(HashSet::new));

            keysNotInLab = new HashSet<>(tankerKeys);
            keysNotInLab.removeAll(segmentKeys);

        }

        var keysNotInTanker = new HashSet<>(segmentKeys);
        keysNotInTanker.removeAll(tankerKeys);

        result.put("keys_not_in_lab", keysNotInLab);
        result.put("keys_not_in_tanker", keysNotInTanker);

        return new ObjectMapper().convertValue(result, JsonNode.class);
    }

    @Override
    public Map<String, Translations> deleteSegmentKeysFromTanker(String segmentId) {
        return withSqlTransaction(tables -> {
            Segment segment = get(segmentId);

            String nameKey = segment.getTankerNameKey();
            String descriptionKey = segment.getTankerDescriptionKey();

            Translations name = getTranslationsFromTankerByKey(nameKey);
            Translations description = getTranslationsFromTankerByKey(descriptionKey);
            Map<String, Translations> translations = new HashMap<>();
            translations.put("name", name);
            translations.put("description", description);

            Tanker tanker = getTanker();
            tanker.deleteKeyFromTanker(nameKey);
            tanker.deleteKeyFromTanker(descriptionKey);

            updateSegment(segmentId, updated -> updated.setTankerNameKey("").setTankerDescriptionKey(""));
            LOG.debug(
                    String.format("Tanker keys of segment %s %s removed from Tanker and from database",
                            segment.getId(),
                            segment.getName())
            );

            return translations;
        });
    }

    @Override
    public Map<Long, SegmentName> getSegmentNamesWithExportId(Long keywordId) {
        Map<Long, SegmentName> mapping = new HashMap<>();
        tables()
                .segments()
                .selectNameWithExportId(keywordId)
                .fetch()
                .forEach(record -> {
                    long exportId = record.get(SegmentExportsTable.EXPORT_SEGMENT_ID);
                    mapping.putIfAbsent(
                            exportId,
                            new SegmentName(record.get(SegmentsTable.NAME_EN), record.get(SegmentsTable.NAME_RU))
                    );
                });

        return mapping;
    }

    private <T> T withSegmentForUpdate(String id, BiFunction<Tables, Segment.Builder, T> updater) {
        return withSqlTransaction(tables -> {
            Segment.Builder segment = fetchSegmentForUpdate(id, tables);
            return updater.apply(tables, segment);
        });
    }

    private Segment updateSegment(String id, Consumer<Segment.Builder> updater) {
        return withSegmentForUpdate(id, (tables, segment) -> {
            updater.accept(segment);
            tables.segments().updateQuery(segment.build()).execute();
            return localizeSegment(segment);
        });
    }

    private <T> T withSegmentGroupForUpdate(String id, BiFunction<Tables, SegmentGroup.Builder, T> updater) {
        return withSqlTransaction(tables -> {
            SegmentGroup.Builder group = fetchSegmentGroupForUpdate(id, tables).toBuilder();
            return updater.apply(tables, group);
        });
    }

    private SegmentGroup updateGroup(String id, Consumer<SegmentGroup.Builder> updater) {
        return withSegmentGroupForUpdate(id, (tables, group) -> {
            updater.accept(group);
            tables.segments().updateSegmentGroupQuery(group.build()).execute();
            return localizeGroup(group);
        });
    }

    private SegmentGroup fetchSegmentGroup(String id, Tables tables, boolean localized) {
        RecordMapper<Record, SegmentGroup> recordMapper;

        if (localized) {
            recordMapper = this::readGroupLocalized;
        } else {
            recordMapper = SegmentsTable::readGroup;
        }

        return tables.segments().selectGroupByIdQuery(id)
                .fetchOptional(recordMapper)
                .orElseThrow(Exceptions::notFound);
    }

    private SegmentGroup fetchSegmentGroupForUpdate(String id, Tables tables) {
        SegmentsTable segments = tables.segments();
        return segments
                .selectGroupByIdQuery(id)
                .fetchOptional(SegmentsTable::readGroup)
                .orElseThrow(Exceptions::notFound);
    }

    private Segment fetchSegment(String id, Tables tables) {
        return tables
                .segments()
                .selectByIdQuery(id)
                .fetchOptional(this::readSegmentLocalized)
                .orElseThrow(NotFoundException::new);
    }

    private Segment fetchSegmentAllTypes(String id, Tables tables) {
        return tables
                .segments()
                .selectByIdAllTypesQuery(id)
                .fetchOptional(this::readSegmentLocalized)
                .orElseThrow(NotFoundException::new);
    }

    private Segment fetchRawSegment(String id, Tables tables) {
        return tables
                .segments()
                .selectByIdQuery(id)
                .fetchOptionalInto(Segment.class)
                .orElseThrow(NotFoundException::new);
    }

    private Segment.Builder fetchSegmentForUpdate(String id, Tables tables) {
        return tables
                .segments()
                .selectByIdQuery(id)
                .fetchOptional(tables.segments()::readSegment)
                .orElseThrow(NotFoundException::new)
                .toBuilder();
    }

    private void storeResponsiblesAndStakeholders(Segment segment, Tables tables) {
        String id = segment.getId();
        long created = Instant.now().getEpochSecond();
        long modified = created;

        ResponsiblesTable responsiblesTable = tables.responsibles();
        segment.getResponsiblesList().forEach(
                responsible -> responsiblesTable.insertQuery(responsible, id, created, modified).execute()
        );
        StakeholdersTable stakeholdersTable = tables.stakeholders();
        segment.getStakeholdersList().forEach(
                stakeholder -> stakeholdersTable.insertQuery(stakeholder, id, created, modified).execute()
        );
    }

    private void storeSegmentReferences(Segment segment, Tables tables) {
        modelSegmentRelations(tables).store(segment);
        storeResponsiblesAndStakeholders(segment, tables);
    }

    private ModelSegmentRelations modelSegmentRelations(Tables tables) {
        return new ModelSegmentRelations(tables);
    }

    @Override
    public SegmentIdPair getSegmentPairBySegmentLabId(String segmentLabId) {
        return tables().segmentIds()
                .selectBySegmentIdQuery(segmentLabId)
                .fetchOptional(SegmentIdsTable::readSegmentIdPair)
                .orElseThrow(Exceptions::notFound)
                .toBuilder()
                .build();
    }

    @Override
    public List<SegmentIdPair> updateSegmentIdsTable(YPath sourceTable) {
        try {
            ListF<YTreeMapNode> result = Cf.arrayList();
            yt.getHahn().tables().read(
                    sourceTable,
                    YTableEntryTypes.YSON,
                    (Consumer<YTreeMapNode>) result::add);

            List<SegmentIdPair> idsPairs = result.map(record -> SegmentIdPair.newBuilder()
                    .setSegmentLabId(record.getString("segment_lab_id"))
                    .setUserSetId(Long.toUnsignedString(record.getLong("user_set_id")))
                    .build());

            return withSqlTransaction(tables -> {
                tables.segmentIds().truncateTableQuery().execute();
                tables.segmentIds().batchInsertQuery(idsPairs).execute();

                return tables.segmentIds().selectQuery().fetchInto(SegmentIdPair.class);
            });
        } catch (Exception e) {
            LOG.error("Error storing segment ids to PGaaS: {}", e.toString());
        }

        return null;
    }

    @Override
    public DefaultSegmentService clone() {
        return new DefaultSegmentService(environment(), sql(), tankerConfig, siberiaClient, yt, lab, audience);
    }

}
