import logging
import tvmauth

import library.python.resource as rs

from crypta.lab.lib import common
import crypta.lib.python.audience.client as audience
from crypta.lib.python.bt import workflow
from crypta.lib.python.bt.conf import conf
from crypta.lib.python.bt.tasks import (
    YtTask,
    YQLTaskV1 as YQLTask,
)

try:
    from cStringIO import StringIO
except ImportError:
    from io import StringIO

logger = logging.getLogger(__name__)


class WithPrivateAudienceClient(object):
    @property
    def audience_client(self):
        tvm_client = tvmauth.TvmClient(tvmauth.TvmApiClientSettings(
            self_tvm_id=conf.proto.Audience.SourceTvmId,
            self_secret=conf.proto.Audience.Secret,
            dsts=[conf.proto.Audience.DestinationTvmId],
        ))

        return audience.PrivateApiAudienceClient(
            tvm_client=tvm_client,
            tvm_dst_id=conf.proto.Audience.DestinationTvmId,
            logger=logger,
        )


def download_table_content(yt, table, table_field):
    n_rows = yt.row_count(table)
    logger.info('Going to download {} records from {}'.format(n_rows, table))

    data = StringIO()
    for i, row in enumerate(yt.read_table(yt.TablePath(table, columns=[table_field]))):
        if not (i % round(float(n_rows) * 0.05)):
            logger.info('%.1f%%' % (100 * (i + 1) / float(n_rows)))
        data.write(str(row[table_field]) + '\n')

    logger.info('Downloading finished!')
    return data.getvalue()


class Create(workflow.IndependentTask, YtTask, WithPrivateAudienceClient, common.WithApi, YQLTask):
    login = workflow.Parameter()
    segment_name = workflow.Parameter()
    table = workflow.Parameter()
    table_field = workflow.Parameter()
    audience_lab_id = workflow.Parameter()

    @property
    def query(self):
        return rs.find('/crypta/lab/audience.yql').format(
            input_table=self.table,
            profile=conf.paths.profiles_for_14days,
            table_field=self.table_field,
            output_table=self.filtered_yuids,
        )

    def run(self, **kwargs):
        with self.yt.TempTable() as self.filtered_yuids:
            super(Create, self).run(**kwargs)

            upload_result = self.audience_client.upload_segment_from_data(
                download_table_content(self.yt, self.filtered_yuids, self.table_field),
                segment_name=self.segment_name,
                content_type='yuid',
                hashed=False,
                ulogin=self.login,
            )

        logger.info(upload_result)

        modified_audience = self.api.lab.setExternalId(
            id=self.audience_lab_id,
            externalId=upload_result['id'],
        ).result()
        logger.info(modified_audience)


class Modify(workflow.IndependentTask, YtTask, WithPrivateAudienceClient, YQLTask):
    login = workflow.Parameter()
    segment_id = workflow.Parameter()
    table = workflow.Parameter()
    table_field = workflow.Parameter()
    modification_type = workflow.Parameter(default='replace')

    @property
    def query(self):
        return rs.find('/crypta/lab/audience.yql').format(
            input_table=self.table,
            profile=conf.paths.profiles_for_14days,
            table_field=self.table_field,
            output_table=self.filtered_yuids,
        )

    def run(self, **kwargs):
        with self.yt.TempTable() as self.filtered_yuids:
            super(Modify, self).run(**kwargs)

            modify_result = self.audience_client.modify_segment_with_data(
                download_table_content(self.yt, self.filtered_yuids, self.table_field),
                audience_segment_id=self.segment_id,
                modification_type=self.modification_type,
                ulogin=self.login,
            )

        logger.info(modify_result)


class Delete(workflow.IndependentTask, YtTask, WithPrivateAudienceClient):
    login = workflow.Parameter()
    segment_id = workflow.Parameter()

    def run(self, **kwargs):
        delete_result = self.audience_client.delete_segment(
            audience_segment_id=self.segment_id,
            ulogin=self.login,
        )

        logger.info(delete_result)
