import logging
import time
import uuid

from yt import wrapper

from crypta.lab.lib import tables
from crypta.lab.lib.crypta_id.userdata import PrepareCryptaIDUserData
from crypta.lib.python.bt import workflow
from crypta.lib.python.bt.commons import dates
from crypta.lib.python.bt.conf import conf
from crypta.lib.proto import user_data
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)


class PrepareAllCryptaIDUserData(PrepareCryptaIDUserData):
    day = workflow.Parameter(parse=dates.parse_day)

    @property
    def str_day(self):
        return self.day.strftime('%Y-%m-%d')

    @property
    def userdata(self):
        return conf.paths.lab.data.all_crypta_id.userdata

    @property
    def userdata_stats(self):
        return conf.paths.lab.data.all_crypta_id.userdata_stats

    def postprocess_tables(self, table_to_process):
        self.sort(self.userdata, self.userdata, sort_by=self.id_type)

        tables_proto = {
            self.userdata: user_data.user_data_pb2.TUserData,
            self.userdata_stats: user_data.user_data_stats_pb2.TUserDataStats,
        }

        for table_path, proto_message_type in tables_proto.iteritems():
            self.yt.set_attribute(table_path, table_to_process.Attributes.DATA_ID, str(uuid.uuid4()))
            yt_helpers.set_yql_proto_fields(table_path, proto_message_type, self.yt)
            self.yt.set_attribute(table_path, table_to_process.Attributes.LAST_UPDATE_DATE, str(self.day))

        self.yt.set_attribute(self.userdata, table_to_process.Attributes.LAST_UPDATE_TIMESTAMP, int(time.time()))

    def run(self, **kwargs):
        cryptaid_regions = wrapper.TablePath(
            name=conf.paths.ids_storage.crypta_id.regions,
            columns=('main_region_city', 'main_region_country', 'crypta_id'),
        )

        apps = conf.paths.ids_storage.crypta_id.apps
        vectors = conf.paths.vectors.crypta_id.all_monthly
        profiles = conf.paths.cryptaid_profiles_for_14days
        devices = conf.paths.ids_storage.crypta_id.devices
        cids_words_weights = conf.paths.affinitives.crypta_id.words
        cids_hosts_weights = conf.paths.affinitives.crypta_id.hosts

        sources = [vectors, cryptaid_regions, profiles, devices, cids_words_weights, cids_hosts_weights, apps]

        mapper_files = [self.yt.TablePath(
            conf.paths.ids_storage.crypta_id.apps_weights,
            attributes={
                'format': 'yson',
                'file_name': 'apps_weights',
            }
        )]

        self.run_map_reduce(sources, vectors, profiles, mapper_files=mapper_files)
        self.postprocess_tables(table_to_process=tables.UserData)
