#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import os
import time

import library.python.resource as resource

import crypta.lib.python.bt.conf.conf as conf
import crypta.lib.python.bt.commons.dates as dates
from crypta.lab.lib.common import WithApi
from crypta.lib.python.bt.tasks import YQLTaskV1 as YQLTask
from crypta.lib.python.bt.workflow import IndependentTask, Parameter
from crypta.lib.python.bt.workflow.targets.table import HasAttribute


logger = logging.getLogger(__name__)


class PrepareRegions(YQLTask, IndependentTask):
    day = Parameter(parse=dates.parse_day)

    @property
    def _attribute(self):
        return 'generate_date'

    @property
    def destination(self):
        return conf.paths.ids_storage.crypta_id.regions

    def targets(self):
        yield HasAttribute(
            self.yt,
            self.destination,
            self._attribute,
            self.day.isoformat(),
        )

    @property
    def attached_files(self):
        return [
            {
                'name': 'aggregation_lib.sql',
                'content': '/lib/aggregation_lib.sql',
                'disposition': 'resource',
                'type': 'library',
            },
            {
                'name': 'metrica_lib.sql',
                'content': '/lib/metrica_lib.sql',
                'disposition': 'resource',
                'type': 'library',
            },
        ]

    @property
    def query(self):
        return resource.find('/crypta/lab/crypta_id/regions.yql').format(
            yuid_with_all_info=conf.paths.ids_storage.yandexuid.yuid_with_all,
            app_metrica_month=conf.paths.ids_storage.device_id.app_metrica_month,
            vertices_no_multi_profile=conf.paths.graph.vertices_no_multi_profile,
            output_table=self.destination,
        )

    def run(self, **kwargs):
        super(PrepareRegions, self).run(**kwargs)
        self.yt.set_attribute(self.destination, self._attribute, self.day.isoformat())


class PrepareRegionsMonitoring(YQLTask, WithApi):
    day = Parameter(parse=dates.parse_day)

    @property
    def query(self):
        return resource.find('/crypta/lab/crypta_id/regions_metrics.yql').format(
            crypta_id_regions=conf.paths.ids_storage.crypta_id.regions,
            taxi_data=conf.paths.taxi_user_profile,
            beru_data=conf.paths.beru_regions,
            regions_mapping=conf.paths.regions_info,
            vertices_no_multi_profile=conf.paths.graph.vertices_no_multi_profile,
            output_table=self.destination,
        )

    def requires(self):
        yield PrepareRegions(day=self.day)

    @property
    def _attribute(self):
        return 'generate_date'

    @property
    def destination(self):
        return conf.paths.lab.data.crypta_id.region_stats

    def targets(self):
        yield HasAttribute(
            self.yt,
            self.destination,
            self._attribute,
            self.day.isoformat(),
        )

    def report_metric(self, name, value):
        reported_event = self.api.metric.report(
            name=name,
            hostname='crypta_profile_{}'.format(os.getenv('CRYPTA_ENVIRONMENT', 'develop').lower()),
            group='crypta_id_region',
            value=value,
            timestamp=int(time.mktime(self.day.timetuple())),
        ).result()

        logger.info('Reported %s', reported_event)

    def run(self, **kwargs):
        super(PrepareRegionsMonitoring, self).run(**kwargs)

        for row in self.yt.read_table(self.destination):
            self.report_metric('{}.{}.count'.format(row['source'], 'correct'), row['correct_count'])
            self.report_metric('{}.{}.count'.format(row['source'], 'total'), row['total_count'])

        self.yt.set_attribute(self.destination, self._attribute, self.day.isoformat())
