import logging
import os

from crypta.lab.lib import tables
from crypta.lab.lib.userdata import PrepareUserData
from crypta.lib.proto import user_data
from crypta.lib.python import templater
from crypta.lib.python.bt import (
    tasks,
    workflow,
)
from crypta.lib.python.bt.commons import dates
from crypta.lib.python.bt.conf import conf
from crypta.lib.python.bt.workflow.targets import table as table_targets
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)


class PrepareMixedUserData(PrepareUserData):
    day = workflow.Parameter(parse=dates.parse_day)

    def requires(self):
        yield tasks.ExternalInputWithAttribute(
            yt_table=self.userdata,
            attribute=tables.UserData.Attributes.LAST_UPDATE_DATE,
            value=str(self.day),
        )
        yield tasks.ExternalInputWithAttribute(
            yt_table=self.cryptaid_userdata,
            attribute=tables.UserData.Attributes.LAST_UPDATE_DATE,
            value=str(self.day),
        )

    def targets(self):
        yield table_targets.HasAttribute(
            yt=self.yt,
            table=self.mixed_userdata,
            attribute=tables.UserData.Attributes.LAST_UPDATE_DATE,
            value=str(self.day),
        )

    @property
    def userdata(self):
        return conf.paths.lab.data.userdata

    @property
    def cryptaid_userdata(self):
        return conf.paths.lab.data.crypta_id.userdata

    @property
    def mixed_userdata(self):
        return conf.paths.lab.data.mixed.userdata

    def mix_userdata(self):
        query = templater.render_resource(
            '/crypta/lab/mix_userdata.yql',
            strict=True,
            vars={
                'userdata_by_yandexuid_table': self.userdata,
                'userdata_by_cryptaid_table': self.cryptaid_userdata,
                'mixed_userdata_table': self.mixed_userdata,
            },
        )
        self.yql_client.execute(
            query=query,
            transaction=str(self.transaction_id),
            title='YQL Mix userdata by yandexuid and userdata by cryptaid',
        )

    def run(self, **kwargs):
        self.mix_userdata()
        if os.environ.get('CRYPTA_ENVIRONMENT') == 'production':
            assert self.yt.row_count(self.userdata) == self.yt.row_count(self.mixed_userdata), \
                'Mixed UserData should contain the same row count as UserData by yandexuid'
        yt_helpers.set_yql_proto_fields(self.mixed_userdata, user_data.user_data_pb2.TUserData, self.yt)
        self.yt.set_attribute(self.mixed_userdata, tables.UserData.Attributes.LAST_UPDATE_DATE, str(self.day))
