#pragma once

#include <crypta/lib/proto/user_data/user_data_stats.pb.h>

#include <util/generic/map.h>

template <>
struct TLess<NLab::TSegment> {
    bool operator()(const NLab::TSegment& lhs, const NLab::TSegment& rhs) const {
        const auto lhsTie = std::make_tuple(lhs.keyword(), lhs.id());
        const auto rhsTie = std::make_tuple(rhs.keyword(), rhs.id());
        return lhsTie < rhsTie;
    }
};

namespace NLab {
    struct TSegmentsAccessor {
        using TValue = TSegment;
        using TCountedValue = TUserDataStats::TSegmentCount;

        static const TValue& GetValue(const TCountedValue& countedValue) {
            return countedValue.GetSegment();
        }

        static void SetValue(TCountedValue* countedValue, const TValue& value) {
            return countedValue->MutableSegment()->CopyFrom(value);
        }

        static auto* AddCountedValue(TUserDataStats::TStrataStats& strataStats) {
            return strataStats.AddSegment();
        }
    };

    struct TAgeAccessor {
        using TValue = TAge;
        using TCountedValue = TAgeCount;

        static TValue GetValue(const TCountedValue& countedValue) {
            return countedValue.GetAge();
        }

        static void SetValue(TCountedValue* countedValue, const TValue& value) {
            return countedValue->SetAge(value);
        }

        static auto* AddCountedValue(TUserDataStats::TStrataStats& strataStats) {
            return strataStats.AddAge();
        }
    };

    struct TGenderAccessor {
        using TValue = TGender;
        using TCountedValue = TGenderCount;

        static TValue GetValue(const TCountedValue& countedValue) {
            return countedValue.GetGender();
        }

        static void SetValue(TCountedValue* countedValue, const TValue& value) {
            return countedValue->SetGender(value);
        }

        static auto* AddCountedValue(TUserDataStats::TStrataStats& strataStats) {
            return strataStats.AddGender();
        }
    };

    struct TIncomeAccessor {
        using TValue = TIncome;
        using TCountedValue = TIncomeCount;

        static TValue GetValue(const TCountedValue& countedValue) {
            return countedValue.GetIncome();
        }

        static void SetValue(TCountedValue* countedValue, const TValue& value) {
            return countedValue->SetIncome(value);
        }

        static auto* AddCountedValue(TUserDataStats::TStrataStats& strataStats) {
            return strataStats.AddIncome();
        }
    };

    template<typename TAccessor>
    class TCountedAggregator {
    public:
        void UpdateWith(const typename TAccessor::TCountedValue& countedValue) {
            const auto& additionalCount = countedValue.GetCount();
            Counts[TAccessor::GetValue(countedValue)] += additionalCount;
        }

        void MergeInto(TUserDataStats::TStrataStats& strataStats) {
            for (const auto& [value, count] : Counts) {
                auto* countedValue = TAccessor::AddCountedValue(strataStats);

                TAccessor::SetValue(countedValue, value);
                countedValue->SetCount(count);
            }
        }

    private:
        TMap<typename TAccessor::TValue, ui64> Counts;
    };

    using TSegmentsAggregator = TCountedAggregator<TSegmentsAccessor>;
    using TAgeAggregator = TCountedAggregator<TAgeAccessor>;
    using TGenderAggregator = TCountedAggregator<TGenderAccessor>;
    using TIncomeAggregator = TCountedAggregator<TIncomeAccessor>;
}
