#include <crypta/lab/lib/native/describe.h>
#include <crypta/lib/native/identifiers/lib/generic.h>
#include <library/cpp/digest/md5/md5.h>
#include <util/generic/hash_set.h>
#include <util/string/ascii.h>

using namespace NLab;

constexpr const char* GROUP_ID = "GroupID";

void TValidateMapper::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    const auto identifierType = ConvertLabType(State->GetSource().GetIdType());
    const auto identifierKey = State->GetSource().GetKey();

    for (; input->IsValid(); input->Next()) {
        const auto row = input->GetRow();
        if (!row.HasKey(identifierKey)) {
            output->AddRow(row);
            continue;
        }
        const auto nodeId = row[identifierKey];
        if (nodeId.IsNull()) {
            output->AddRow(row);
            continue;
        }
        const auto identifier = nodeId.ConvertTo<TString>();
        if (!identifierType) {
            output->AddRow(row);
            continue;
        }
        NIdentifiers::TGenericID genericId(identifierType.GetRef(), identifier);
        if (!genericId.IsValid()) {
            output->AddRow(row);
        }
    }
}

void TRenameIdentifierMapper::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    const auto sourceKey = State->GetSource().GetKey();
    const auto destinationKey = State->GetDestination().GetKey();
    const auto groupingKey = State->GetSample().Getgrouping_key();

    for (; input->IsValid(); input->Next()) {
        auto row = input->GetRow();

        auto id = row[sourceKey];
        row.AsMap().erase(sourceKey);
        if (id.IsNull() || id.IsUndefined()) {
            continue;
        }
        row[destinationKey] = id.ConvertTo<TString>();
        if (groupingKey.empty()) {
            row[GROUP_ID] = TNode::CreateEntity();
        } else {
            auto groupingKeyValue = row[groupingKey];
            if ((!groupingKeyValue.IsNull() && !groupingKeyValue.IsUndefined())) {
                auto groupId = groupingKeyValue.ConvertTo<TString>();
                row[GROUP_ID] = groupId;
            }
        }
        output->AddRow(row);
    }
}

void TGenericJoinReducer::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    TVector<TNode> allLhs{};
    for (; input->IsValid(); input->Next()) {
        if (input->GetTableIndex() == 0) {
            allLhs.push_back(input->GetRow());
            if (allLhs.size() > State->GetForeignTableMaxRecords()) {
                return;
            }
        }
        if (input->GetTableIndex() == 1) {
            for (const auto& lhs : allLhs) {
                auto rhs = input->GetRow();
                auto& rhsMap = rhs.AsMap();
                if (State->GetMode() == NLab::TJoinOptions_EMode_M_KEEP_BOTH) {
                    for (const auto& it : lhs.AsMap()) {
                        if (!rhsMap.contains(it.first)) {
                            rhsMap[it.first] = it.second;
                        }
                    }
                }
                output->AddRow(rhs);
            }
        }
    }
}

void TSubsamplesJoinReducer::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    TString groupName;
    TNode subsample;
    int outputIndex = 0;

    for (; input->IsValid(); input->Next()) {
        const auto& tableIndex = input->GetTableIndex();

        if (tableIndex == 0) {
            outputIndex = input->GetRowIndex();
            const auto& subsample_info = input->GetRow();
            groupName = subsample_info[State->GetGroupName()].ConvertTo<TString>();
        } else if (tableIndex == 1) {
            subsample = input->GetRow();

            if (subsample[State->GetGroupName()].ConvertTo<TString>() == groupName) {
                output->AddRow(subsample, outputIndex);
            }
        }
    }
};

void TIdentityMapper::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    for (; input->IsValid(); input->Next()) {
        output->AddRow(input->MoveRow());
    }
}

void TUniqueReducer::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    for (; input->IsValid(); input->Next()) {
        output->AddRow(input->MoveRow());
        return;
    }
}

void TAddGroupingKey::Do(TTableReader<TNode>* input, TTableWriter<TNode>* output) {
    const auto groupingKey = State->GetSample().Getgrouping_key();

    for (; input->IsValid(); input->Next()) {
        auto row = input->GetRow();
        if (groupingKey.empty()) {
            row[GROUP_ID] = TNode::CreateEntity();
        } else {
            if (!row.HasKey(groupingKey)) {
                row[GROUP_ID] = TNode::CreateEntity();
            } else {
                auto groupId = row[groupingKey];
                if (groupId.IsNull()) {
                    row[GROUP_ID] = TNode::CreateEntity();
                } else {
                    row[GROUP_ID] = groupId.ConvertTo<TString>();
                }
            }
        }
        output->AddRow(row);
    }
}
