#include <google/protobuf/util/message_differencer.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>
#include <mapreduce/yt/tests/yt_unittest_lib/yt_unittest_lib.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <crypta/lab/lib/native/extract_userdata.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/proto/user_data/user_data.pb.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>

using namespace NLab;
using namespace NYT;
using namespace NYT::NTesting;

Y_UNIT_TEST_SUITE(TExtractUserData) {

    Y_UNIT_TEST(ProcessProfilesTest) {
        JoblessInitialize();

        const auto topCommonSiteIds = TNode::CreateList().Add(100001ul);

        const auto affinitiveSiteIds = TNode::CreateMap()
            ("200001ul", 1.05);

        const auto exactSocdem = TNode::CreateMap()
            ("gender", "f");


        const auto lalInternal = TNode::CreateMap()
            ("1304", 0.6)
            ("2257", 1);

        const auto lalFake = TNode::CreateMap()
            ("1765", 0.6)
            ("2076", 0.4);

        const auto heuristicInternal = TNode::CreateList().Add(1654);

        TNode row = TNode()
            ("crypta_id", 123ul)
            ("affinitive_site_ids", affinitiveSiteIds)
            ("top_common_site_ids", topCommonSiteIds)
            ("exact_socdem", exactSocdem)
            ("lal_internal", lalInternal)
            ("heuristic_internal", heuristicInternal)
            ("lal_fake", lalFake);

        TUserData resUserData;
        TExtractUserDataState::TProfilesLogState state;

        auto& keyword_fields = *state.MutableKeywordFields();
        keyword_fields[546].AddField("lal_internal");
        keyword_fields[549].AddField("heuristic_internal");
        keyword_fields[666].AddField("lal_fake");

        const TString refUserDataYaml = "segments:\n"
                                        "   segment:\n"
                                        "       - keyword: 546\n"
                                        "         i_d: 1304\n"
                                        "         score: 0.6\n"
                                        "       - keyword: 546\n"
                                        "         i_d: 2257\n"
                                        "         score: 1\n"
                                        "       - keyword: 549\n"
                                        "         i_d: 1654\n"
                                        "       - keyword: 666\n"
                                        "         i_d: 1765\n"
                                        "       - keyword: 666\n"
                                        "         i_d: 2076\n";


        const auto& refUserData = NCrypta::Yaml2Proto<TUserData>(refUserDataYaml);

        TExtractUserData::ProcessProfilesLog(row, resUserData, state);

        UNIT_ASSERT_EQUAL(1, resUserData.GetAffinities().GetTopCommonSites().TokenSize());
        UNIT_ASSERT_EQUAL(1, resUserData.GetAffinities().GetAffinitiveSites().TokenSize());
        UNIT_ASSERT_EQUAL(FEMALE, resUserData.GetAttributes().GetGender());

        google::protobuf::util::MessageDifferencer differencer;
        differencer.set_repeated_field_comparison(google::protobuf::util::MessageDifferencer::RepeatedFieldComparison::AS_SET);
        UNIT_ASSERT(differencer.Compare(refUserData.GetSegments(), resUserData.GetSegments()));
    }

    Y_UNIT_TEST(ProcessRegionsAndDevice) {
        JoblessInitialize();

        TNode moscowRow = TNode()
            ("yuid", "123")
            ("ip_activity_type", "active")
            ("main_region_city", static_cast<i32>(213))
            ("main_region_country", static_cast<i32>(225));

        TNode otherRow = TNode()
            ("yuid", "123")
            ("ip_activity_type", "active")
            ("main_region_city", static_cast<i32>(915))
            ("main_region_country", static_cast<i32>(225));

        TNode cidRow = TNode()
            ("crypta_id", "123")
            ("main_region_city", static_cast<i32>(915))
            ("main_region_country", static_cast<i32>(225));

        TUserData userData;
        TExtractUserData::ProcessRegionsAndDevice(moscowRow, userData, TId::EIdType::Yandexuid);

        UNIT_ASSERT_EQUAL(userData.GetAttributes().GetRegion(), 213ul);
        UNIT_ASSERT_EQUAL(userData.GetAttributes().GetCountry(), RUSSIA);

        userData.Clear();
        TExtractUserData::ProcessRegionsAndDevice(otherRow, userData, TId::EIdType::Yandexuid);

        UNIT_ASSERT_EQUAL(userData.GetAttributes().GetRegion(), 915ul);
        UNIT_ASSERT_EQUAL(userData.GetAttributes().GetCountry(), RUSSIA);

        userData.Clear();
        TExtractUserData::ProcessRegionsAndDevice(cidRow, userData, TId::EIdType::CryptaId);
        UNIT_ASSERT_EQUAL(userData.GetAttributes().GetRegion(), 915ul);
        UNIT_ASSERT_EQUAL(userData.GetAttributes().GetCountry(), RUSSIA);
    }

    Y_UNIT_TEST(ProcessApps) {
        JoblessInitialize();

        const auto apps = TNode::CreateList().Add("com.some.app").Add("ru.another.app");

        TNode row = TNode()
            ("crypta_id", "123")
            ("apps", apps);

        const TString refUserDataYaml = "affinities:\n"
                                        "   apps:\n"
                                        "       token:\n"
                                        "       - token: com.some.app\n"
                                        "         weight: 223\n"
                                        "       - token: ru.another.app\n"
                                        "         weight: 1345678\n";


        const auto& refUserData = NCrypta::Yaml2Proto<TUserData>(refUserDataYaml);
        TUserData resUserData;
        TExtractUserData mapper;
        mapper.AppsWeights = {{"com.some.app", 223}, {"ru.another.app", 1345678}};

        mapper.ProcessApps(row, resUserData);

        google::protobuf::util::MessageDifferencer differencer;
        differencer.set_repeated_field_comparison(google::protobuf::util::MessageDifferencer::RepeatedFieldComparison::AS_SET);
        UNIT_ASSERT(differencer.Compare(refUserData.GetAffinities().GetApps(), resUserData.GetAffinities().GetApps()));
    }
}
