#include <google/protobuf/util/message_differencer.h>
#include <library/cpp/testing/unittest/registar.h>

#include <crypta/lab/lib/native/segments_user_data_stats_aggregator.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/proto/user_data/user_data.pb.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>

using namespace NLab;

namespace {
    TUserData GetUserDataWoAffinities() {
        TUserData userData;

        userData.SetGroupID("Group");

        auto* attributes = userData.MutableAttributes();
        attributes->SetAge(TAge::FROM_0_TO_17);
        attributes->SetDevice(TDevice::DESKTOP);
        attributes->SetGender(TGender::MALE);
        attributes->SetRegion(100);
        attributes->SetIncome(TIncome::INCOME_B1);
        attributes->SetCountry(TCountry::RUSSIA);
        attributes->SetCity(TCity::MOSCOW);
        attributes->SetHasCryptaID(true);

        auto* vectors = userData.MutableVectors();
        auto* vector = vectors->MutableVector();
        vector->AddData(1);
        vector->AddData(2);

        auto* segment = userData.MutableSegments()->AddSegment();
        segment->SetKeyword(1);
        segment->SetID(2);

        auto* identifiers = userData.MutableIdentifiers();
        identifiers->SetNotUnique(false);
        identifiers->MutableIdentifiers()->insert({"str_1", "str_2"});

        userData.SetWithoutData(false);
        userData.SetCryptaID("crypta_id");

        return userData;
    }

    TUserData GetUserDataWithAffinities() {
        auto userData = GetUserDataWoAffinities();

        auto* affinities = userData.MutableAffinities();
        auto* word = affinities->MutableWords()->AddToken();
        word->SetToken("word_1");
        word->SetWeight(1.1);

        auto* host = affinities->MutableHosts()->AddToken();
        host->SetToken("host_1");
        host->SetWeight(1.2);

        auto* app = affinities->MutableApps()->AddToken();
        app->SetToken("app_1");
        app->SetWeight(2.5);

        auto* affinitiveSite = affinities->MutableAffinitiveSites()->AddToken();
        affinitiveSite->SetToken("affinitive_site_1");
        affinitiveSite->SetWeight(1.3);

        auto* topCommonSite = affinities->MutableTopCommonSites()->AddToken();
        topCommonSite->SetToken("top_common_site_1");
        topCommonSite->SetWeight(1.4);

        return userData;
    }

    TUserData GetUserDataWithAffinitiesEncoded() {
        auto userData = GetUserDataWoAffinities();

        auto* affinitiesEncoded = userData.MutableAffinitiesEncoded();
        affinitiesEncoded->MutableWords()->Add(1);
        affinitiesEncoded->MutableHosts()->Add(2);
        affinitiesEncoded->MutableApps()->Add(3);

        auto* affinitiveSite = affinitiesEncoded->MutableAffinitiveSites()->Add();
        affinitiveSite->SetId(3);
        affinitiveSite->SetWeight(1.3);

        auto* topCommonSite = affinitiesEncoded->MutableTopCommonSites()->Add();
        topCommonSite->SetId(4);
        topCommonSite->SetWeight(1.4);

        return userData;
    }
}

Y_UNIT_TEST_SUITE(TSegmentsUserDataStatsAggregator) {
    Y_UNIT_TEST(ConvertUserDataToUserDataStats) {
        const auto& userData = GetUserDataWithAffinities();

        TString refYaml = "attributes:\n"
                          "  age:\n"
                          "  - age: from_0_to_17\n"
                          "    count: 1\n"
                          "  device:\n"
                          "  - device: desktop\n"
                          "    count: 1\n"
                          "  gender:\n"
                          "  - gender: male\n"
                          "    count: 1\n"
                          "  region:\n"
                          "  - region: 100\n"
                          "    count: 1\n"
                          "  income:\n"
                          "  - income: income_b1\n"
                          "    count: 1\n"
                          "  gender_age_income:\n"
                          "  - gender_age_income:\n"
                          "      gender: male\n"
                          "      age: from_0_to_17\n"
                          "      income: income_b1\n"
                          "    count: 1\n"
                          "stratum:\n"
                          "  strata:\n"
                          "  - strata:\n"
                          "      device: desktop\n"
                          "      country: russia\n"
                          "      city: moscow\n"
                          "      has_crypta_i_d: true\n"
                          "    segment:\n"
                          "    - segment:\n"
                          "        keyword: 1\n"
                          "        i_d: 2\n"
                          "      count: 1\n"
                          "    age:\n"
                          "    - age: from_0_to_17\n"
                          "      count: 1\n"
                          "    gender:\n"
                          "    - gender: male\n"
                          "      count: 1\n"
                          "    income:\n"
                          "    - income: income_b1\n"
                          "      count: 1\n"
                          "    count: 1\n"
                          "distributions:\n"
                          "  main:\n"
                          "    mean:\n"
                          "      data:\n"
                          "      - 1\n"
                          "      - 2\n"
                          "    count: 1\n"
                          "counts:\n"
                          "  with_data: 1\n"
                          "  total: 1\n"
                          "  uniq_yuid: 1\n"
                          "group_i_d: Group\n"
                          "identifiers:\n"
                          "  identifiers:\n"
                          "  - key: str_1\n"
                          "    value: str_2\n"
                          "  not_unique: false\n"
                          "affinities:\n"
                          "  hosts:\n"
                          "    token:\n"
                          "    - token: host_1\n"
                          "      weight: 1.2\n"
                          "      count: 1\n"
                          "    tokens_count: 1\n"
                          "    users_count: 1\n"
                          "  words:\n"
                          "    token:\n"
                          "    - token: word_1\n"
                          "      weight: 1.1\n"
                          "      count: 1\n"
                          "    tokens_count: 1\n"
                          "    users_count: 1\n"
                          "  apps:\n"
                          "    token:\n"
                          "    - token: app_1\n"
                          "      weight: 2.5\n"
                          "      count: 1\n"
                          "    tokens_count: 1\n"
                          "    users_count: 1";

        const auto& ref = NCrypta::Yaml2Proto<TUserDataStats>(refYaml);

        NNativeYT::TProtoState<NLab::TUserDataStatsOptions> state;
        TSegmentsUserDataStatsAggregator aggregator(state);

        TUserDataStats userDataStats;
        aggregator.ConvertUserDataToUserDataStats(userDataStats, userData);
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(ref, userDataStats));
    }

    Y_UNIT_TEST(ConvertUserDataToUserDataStatsEncoded) {
        const auto& userData = GetUserDataWithAffinitiesEncoded();

        TString refYaml = "attributes:\n"
                          "  age:\n"
                          "  - age: from_0_to_17\n"
                          "    count: 1\n"
                          "  device:\n"
                          "  - device: desktop\n"
                          "    count: 1\n"
                          "  gender:\n"
                          "  - gender: male\n"
                          "    count: 1\n"
                          "  region:\n"
                          "  - region: 100\n"
                          "    count: 1\n"
                          "  income:\n"
                          "  - income: income_b1\n"
                          "    count: 1\n"
                          "  gender_age_income:\n"
                          "  - gender_age_income:\n"
                          "      gender: male\n"
                          "      age: from_0_to_17\n"
                          "      income: income_b1\n"
                          "    count: 1\n"
                          "stratum:\n"
                          "  strata:\n"
                          "  - strata:\n"
                          "      device: desktop\n"
                          "      country: russia\n"
                          "      city: moscow\n"
                          "      has_crypta_i_d: true\n"
                          "    segment:\n"
                          "    - segment:\n"
                          "        keyword: 1\n"
                          "        i_d: 2\n"
                          "      count: 1\n"
                          "    age:\n"
                          "    - age: from_0_to_17\n"
                          "      count: 1\n"
                          "    gender:\n"
                          "    - gender: male\n"
                          "      count: 1\n"
                          "    income:\n"
                          "    - income: income_b1\n"
                          "      count: 1\n"
                          "    count: 1\n"
                          "distributions:\n"
                          "  main:\n"
                          "    mean:\n"
                          "      data:\n"
                          "      - 1\n"
                          "      - 2\n"
                          "    count: 1\n"
                          "counts:\n"
                          "  with_data: 1\n"
                          "  total: 1\n"
                          "  uniq_yuid: 1\n"
                          "group_i_d: Group\n"
                          "identifiers:\n"
                          "  identifiers:\n"
                          "  - key: str_1\n"
                          "    value: str_2\n"
                          "  not_unique: false\n"
                          "affinities_encoded:\n"
                          "  hosts:\n"
                          "    token:\n"
                          "    - id: 2\n"
                          "      count: 1\n"
                          "    tokens_count: 1\n"
                          "    users_count: 1\n"
                          "  words:\n"
                          "    token:\n"
                          "    - id: 1\n"
                          "      count: 1\n"
                          "    tokens_count: 1\n"
                          "    users_count: 1\n"
                          "  apps:\n"
                          "    token:\n"
                          "    - id: 3\n"
                          "      count: 1\n"
                          "    tokens_count: 1\n"
                          "    users_count: 1";

        const auto& ref = NCrypta::Yaml2Proto<TUserDataStats>(refYaml);

        NNativeYT::TProtoState<NLab::TUserDataStatsOptions> state;
        TSegmentsUserDataStatsAggregator aggregator(state);

        TUserDataStats userDataStats;
        aggregator.ConvertUserDataToUserDataStats(userDataStats, userData);
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(ref, userDataStats));
    }
}
