#include <google/protobuf/util/message_differencer.h>
#include <library/cpp/testing/unittest/registar.h>

#include <crypta/lab/lib/native/user_data_stats_decoder.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>

using namespace NLab;
using namespace NLab::NEncodedUserData;

namespace {
    TString GetUserDataStatsWoAffinitiesStr() {
        return "attributes:\n"
               "  age:\n"
               "  - age: from_0_to_17\n"
               "    count: 1\n"
               "  device:\n"
               "  - device: desktop\n"
               "    count: 1\n"
               "  gender:\n"
               "  - gender: male\n"
               "    count: 1\n"
               "  region:\n"
               "  - region: 100\n"
               "    count: 1\n"
               "  income:\n"
               "  - income: income_b1\n"
               "    count: 1\n"
               "  gender_age_income:\n"
               "  - gender_age_income:\n"
               "      gender: male\n"
               "      age: from_0_to_17\n"
               "      income: income_b1\n"
               "    count: 1\n"
               "stratum:\n"
               "  strata:\n"
               "  - strata:\n"
               "      device: desktop\n"
               "      country: russia\n"
               "      city: moscow\n"
               "      has_crypta_i_d: true\n"
               "    segment:\n"
               "    - segment:\n"
               "        keyword: 1\n"
               "        i_d: 2\n"
               "      count: 1\n"
               "    count: 1\n"
               "distributions:\n"
               "  main:\n"
               "    mean:\n"
               "      data:\n"
               "      - 1.0\n"
               "      - 1.0\n"
               "    count: 1\n"
               "counts:\n"
               "  with_data: 1\n"
               "  total: 1\n"
               "  uniq_yuid: 1\n"
               "group_i_d: Group\n"
               "identifiers:\n"
               "  identifiers:\n"
               "  - key: str_1\n"
               "    value: str_2\n"
               "  not_unique: false\n"
               "segment_info:\n"
               "  info:\n"
               "    str_1: str_2\n";
    }

    TUserDataStats GetUserDataStatsWithAffinitiesEncoded() {
        const TString userDataYaml = GetUserDataStatsWoAffinitiesStr() +
            "affinities_encoded:\n"
            "  hosts:\n"
            "    token:\n"
            "    - id: 1\n"
            "      count: 1\n"
            "    tokens_count: 1\n"
            "    users_count: 1\n"
            "  words:\n"
            "    token:\n"
            "    - id: 1\n"
            "      count: 1\n"
            "    - id: 2\n"
            "      count: 2\n"
            "    - id: 3\n"
            "      count: 3\n"
            "    - id: 4\n"
            "      count: 4\n"
            "    - id: 5\n"
            "      count: 5\n"
            "    tokens_count: 5\n"
            "    users_count: 30\n"
            "  apps:\n"
            "    token:\n"
            "    - id: 1\n"
            "      count: 1\n"
            "    - id: 2\n"
            "      count: 2\n"
            "    tokens_count: 2\n"
            "    users_count: 5";

        return NCrypta::Yaml2Proto<TUserDataStats>(userDataYaml);
    }

    TUserDataStats GetUserDataStatsWithAffinities() {
        const TString userDataYaml = GetUserDataStatsWoAffinitiesStr() +
            "affinities:\n"
            "  hosts:\n"
            "    token:\n"
            "    - token: host_1\n"
            "      weight: 1\n"
            "      count: 1\n"
            "    tokens_count: 1\n"
            "    users_count: 1\n"
            "  words:\n"
            "    token:\n"
            "    - token: word_1\n"
            "      weight: 1\n"
            "      count: 1\n"
            "    - token: word_2\n"
            "      weight: 4\n"
            "      count: 2\n"
            "    - token: word_3\n"
            "      weight: 9\n"
            "      count: 3\n"
            "    - token: word_4\n"
            "      weight: 16\n"
            "      count: 4\n"
            "    - token: word_5\n"
            "      weight: 25\n"
            "      count: 5\n"
            "    tokens_count: 5\n"
            "    users_count: 30\n"
            "  apps:\n"
            "    token:\n"
            "    - token: app_1\n"
            "      weight: 1\n"
            "      count: 1\n"
            "    - token: app_2\n"
            "      weight: 4\n"
            "      count: 2\n"
            "    tokens_count: 2\n"
            "    users_count: 5";

        return NCrypta::Yaml2Proto<TUserDataStats>(userDataYaml);
    }
}

Y_UNIT_TEST_SUITE(TUserDataStatsDecoder) {
    Y_UNIT_TEST(CommonTest) {
        auto userDataStats = GetUserDataStatsWithAffinitiesEncoded();

        TIdToWeightedTokenDict hosts = {
            {1, {.Token = "host_1", .Weight = 1.}}
        };

        TIdToWeightedTokenDict words = {
            {1, {.Token = "word_1", .Weight = 1.}},
            {2, {.Token = "word_2", .Weight = 2.}},
            {3, {.Token = "word_3", .Weight = 3.}},
            {4, {.Token = "word_4", .Weight = 4.}},
            {5, {.Token = "word_5", .Weight = 5.}},
            {6, {.Token = "word_6", .Weight = 6.}},
        };

        TIdToWeightedTokenDict apps = {
            {1, {.Token = "app_1", .Weight = 1.}},
            {2, {.Token = "app_2", .Weight = 2.}},
        };

        NUserDataStatsDecoder::Decode(userDataStats, words, hosts, apps);
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::ApproximatelyEquals(GetUserDataStatsWithAffinities(), userDataStats));
    }

    Y_UNIT_TEST(InvalidDictTest) {
        auto userDataStats = GetUserDataStatsWithAffinitiesEncoded();

        TIdToWeightedTokenDict hosts = {
            {1, {.Token = "host_1", .Weight = 1.}}
        };

        TIdToWeightedTokenDict words = {
            {1, {.Token = "word_1", .Weight = 1.}}
        };

        TIdToWeightedTokenDict apps = {
            {1, {.Token = "app_1", .Weight = 1.}}
        };

        UNIT_ASSERT_EXCEPTION(NUserDataStatsDecoder::Decode(userDataStats, words, hosts, apps), yexception);
    }
}
