import datetime
import logging

import library.python.resource as rs

import crypta.lib.python.bt.conf.conf as conf
from crypta.lib.python.bt.tasks import (
    YQLTaskV1 as YQLTask,
    YtTask,
)
from crypta.lib.python.bt.workflow import (
    IndependentTask,
)
from crypta.lib.python.bt.workflow.targets.table import (
    HasAttribute,
)
import crypta.siberia.bin.common.create_user_set_from_sample_reducer.py as sampler
from crypta.siberia.bin.common.describing.mode.python import describing_mode


logger = logging.getLogger(__name__)


class PrepareSample(YQLTask, IndependentTask):
    @property
    def _today(self):
        return datetime.date.today().isoformat()

    @property
    def _attribute(self):
        return '_day'

    @property
    def destination(self):
        return conf.paths.lab.apps

    @property
    def query(self):
        return rs.find('/crypta/lab/apps_sample.yql').format(
            source=conf.paths.ids_storage.device_id.app_metrica_month,
            destination=self.destination,
            android_count=9000,
            ios_count=1000,
        )

    def targets(self):
        yield HasAttribute(self.yt, self.destination, self._attribute, self._today)

    def run(self, **kwargs):
        super(PrepareSample, self).run(**kwargs)
        self.yt.set_attribute(self.destination, self._attribute, self._today)


class DescribeInSiberia(YtTask):
    def requires(self):
        yield PrepareSample()

    @property
    def _today(self):
        return datetime.date.today().isoformat()

    @property
    def _attribute(self):
        return '_day'

    @property
    def source(self):
        return conf.paths.lab.apps

    @property
    def destination(self):
        return conf.paths.lab.sample_user_sets.apps

    def targets(self):
        yield HasAttribute(self.yt, self.destination, self._attribute, self._today)

    def run(self, **kwargs):
        sampler.create_user_set_from_sample(
            self.yt,
            self.native_map_reduce_with_combiner,
            self.native_map,
            source=self.source,
            destination=self.destination,
            tvm_settings={
                "source_id": conf.proto.Tvm.SourceTvmId,
                "destination_id": conf.proto.Siberia.Tvm.DestinationTvmId,
                "secret": conf.proto.Tvm.Secret,
            },
            group_id_column="app",
            id_type_column="id_type",
            id_column="idfa_gaid",
            sample_size=conf.proto.Options.SampleSize,
            siberia_host=conf.proto.Siberia.Host,
            siberia_port=conf.proto.Siberia.Port,
            max_ids_per_second=conf.proto.Options.MaxIdsPerSecond,
            max_jobs=conf.proto.Options.MaxDescribeJobs,
            describing_mode=describing_mode.SLOW,
            experiment="by_crypta_id",
        )

        self.yt.set_attribute(self.destination, self._attribute, self._today)
