import datetime
import logging

import library.python.resource as rs
import tvmauth

import crypta.lib.python.bt.conf.conf as conf
from crypta.lib.python.bt.tasks import (
    YQLTaskV1 as YQLTask,
    YtTask,
)
from crypta.lib.python.bt.workflow import (
    IndependentTask,
)
from crypta.lib.python.bt.workflow.targets.table import (
    HasAttribute,
)
from crypta.lib.proto.identifiers import id_type_pb2
from crypta.siberia.bin.common import sample_stats_getter
import crypta.siberia.bin.common.create_user_set_from_sample_reducer.py as sampler
from crypta.siberia.bin.common.describing.mode.python import describing_mode
from crypta.siberia.bin.common.siberia_client import SiberiaClient


logger = logging.getLogger(__name__)

MINUTE = 60
HOUR = MINUTE * 60
DAY = HOUR * 24


class PrepareSample(YQLTask, IndependentTask):
    @property
    def _today(self):
        return datetime.date.today().isoformat()

    @property
    def _yesterday(self):
        return (datetime.date.today() - datetime.timedelta(days=1)).isoformat()

    @property
    def _attribute(self):
        return '_day'

    @property
    def destination(self):
        return conf.paths.lab.domains

    @property
    def query(self):
        return rs.find('/crypta/lab/domains_sample.yql').format(
            visitlog="{base}/{day}".format(base=conf.paths.visitlog.daily, day=self._yesterday),
            visitlog_private="{base}/{day}".format(base=conf.paths.visitlog.daily_private, day=self._yesterday),
            top_domains=conf.paths.lab.top_domains,
            destination=self.destination,
            count=10000,
        )

    def targets(self):
        yield HasAttribute(self.yt, self.destination, self._attribute, self._today)

    def run(self, **kwargs):
        super(PrepareSample, self).run(**kwargs)
        self.yt.set_attribute(self.destination, self._attribute, self._today)


class DescribeInSiberia(YtTask):
    def requires(self):
        yield PrepareSample()

    @property
    def _today(self):
        return datetime.date.today().isoformat()

    @property
    def _attribute(self):
        return '_day'

    @property
    def source(self):
        return conf.paths.lab.domains

    @property
    def destination(self):
        return conf.paths.lab.sample_user_sets.domains

    def targets(self):
        yield HasAttribute(self.yt, self.destination, self._attribute, self._today)

    def run(self, **kwargs):
        sampler.create_user_set_from_sample(
            self.yt,
            self.native_map_reduce_with_combiner,
            self.native_map,
            source=self.source,
            destination=self.destination,
            tvm_settings={
                "source_id": conf.proto.Tvm.SourceTvmId,
                "destination_id": conf.proto.Siberia.Tvm.DestinationTvmId,
                "secret": conf.proto.Tvm.Secret,
            },
            group_id_column="hits.domain",
            id_type=id_type_pb2.EIdType.YANDEXUID,
            id_column="hits.yandexuid",
            sample_size=conf.proto.Options.SampleSize,
            siberia_host=conf.proto.Siberia.Host,
            siberia_port=conf.proto.Siberia.Port,
            max_ids_per_second=conf.proto.Options.MaxIdsPerSecond,
            max_jobs=conf.proto.Options.MaxDescribeJobs,
            describing_mode=describing_mode.SLOW,
            experiment="by_crypta_id",
        )

        tvm_client = tvmauth.TvmClient(
            tvmauth.TvmApiClientSettings(
                self_tvm_id=conf.proto.Tvm.SourceTvmId,
                self_secret=conf.proto.Tvm.Secret,
                dsts={"siberia": conf.proto.Siberia.Tvm.DestinationTvmId},
            )
        )

        sample_stats_getter.get_stats(
            self.yt,
            SiberiaClient(conf.proto.Siberia.Host, conf.proto.Siberia.Port),
            tvm_client.get_service_ticket_for("siberia"),
            self.destination,
        )

        self.yt.set_attribute(self.destination, self._attribute, self._today)
