from datetime import timedelta

from crypta.lib.python import crypta_env
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import deploy
from crypta.lib.python.spine.consts import (
    environment,
    dc as dc_consts,
)
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
import crypta.lib.python.spine.deploy.consts as yp_consts
from crypta.lib.python.spine.juggler import juggler_check_generator
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.solomon import solomon_check_generator
from sandbox.projects.crypta import lab
import sandbox.projects.crypta.common.task as crypta_common_task


GRPC_PORT = 80
HTTP_PORT = 81
SOLOMON_PORT = 4242
BALANCER = "rule-estimator.lab.crypta.yandex.net"


class Task(object):
    def __init__(self, task, envs, crit_time, kill_timeout, schedule_interval=None, task_args=None, schedule_daily_start_time=None, retry_interval=None):
        self.task = task
        self.envs = envs
        self.crit_time = crit_time
        self.kill_timeout = kill_timeout
        self.schedule_interval = schedule_interval
        self.task_args = task_args
        self.schedule_daily_start_time = schedule_daily_start_time
        self.retry_interval = retry_interval


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(
        host="crypta-lab",
        tags=["crypta-lab"],
    )

    add_schedulers(juggler)
    add_rule_estimator_api(juggler)
    add_rule_estimator_worker(juggler)
    return juggler


def add_schedulers(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["LAB"])
    all_envs = (environment.STABLE, environment.TESTING)
    prod_envs = (environment.STABLE,)

    for task in (
        Task(
            "crypta.lab.lib.crypta_id.regions.PrepareRegionsMonitoring",
            envs=prod_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_daily_start_time="2020-01-23T05:00:00Z",
            task_args={"day": "yesterday"},
            kill_timeout=timedelta(hours=4),
            retry_interval=timedelta(hours=1),
        ),
        Task(
            "crypta.lab.lib.lookalike.ComputeLookalikeDaily",
            envs=all_envs,
            crit_time=timedelta(days=1),
            schedule_interval=timedelta(hours=2),
            kill_timeout=timedelta(hours=3),
            retry_interval=timedelta(hours=2),
        ),
        Task(
            "crypta.lab.lib.samples.mails_puids_eu.PrepareSample",
            envs=prod_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_daily_start_time="2018-06-09T11:00:00Z",
            kill_timeout=timedelta(hours=3),
            retry_interval=timedelta(hours=2),
        ),
        Task(
            "crypta.lab.lib.samples.domains.DescribeInSiberia",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(hours=1),
            kill_timeout=timedelta(hours=16),
        ),
        Task(
            "crypta.lab.lib.samples.apps.DescribeInSiberia",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(hours=1),
            kill_timeout=timedelta(hours=16),
        ),
        Task(
            "crypta.lab.lib.samples.segments.UploadIdsToDatabase",
            envs=prod_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(hours=1),
            kill_timeout=timedelta(hours=16),
        ),
    ):
        extra_params = {"task_name": task.task}
        if task.task_args:
            extra_params["task_args"] = task.task_args

        for env in task.envs:
            scheduler = sandbox.create_scheduler(
                lab.CryptaLab,
                schedule_interval=task.schedule_interval,
                schedule_daily_start_time=task.schedule_daily_start_time,
                kill_timeout=task.kill_timeout,
                extra_params=extra_params,
                env=env,
                retry_interval=task.retry_interval,
            )
            if env == environment.STABLE:
                scheduler.check(task.crit_time, juggler_service=lab.CryptaLab.get_juggler_service_for_task_name(task.task)).add_yt_dependencies(YtProxy.Group.offline)

    for task in (
        Task(
            "crypta.lab.lib.userdata.PrepareUserData",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(minutes=10),
            kill_timeout=timedelta(hours=4),
            retry_interval=timedelta(minutes=10),
        ),
        Task(
            "crypta.lab.lib.crypta_id.userdata.PrepareCryptaIDUserData",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(minutes=10),
            kill_timeout=timedelta(hours=4),
            retry_interval=timedelta(minutes=10),
        ),
        Task(
            "crypta.lab.lib.crypta_id.all_userdata.PrepareAllCryptaIDUserData",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(minutes=10),
            kill_timeout=timedelta(hours=4),
            retry_interval=timedelta(minutes=10),
        ),
        Task(
            "crypta.lab.lib.mixed_userdata.PrepareMixedUserData",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(minutes=10),
            kill_timeout=timedelta(hours=4),
            retry_interval=timedelta(minutes=10),
        ),
    ):
        for day in ("today", "yesterday"):
            for env in task.envs:
                juggler_service = "CryptaLab_{}_{}".format(task.task, day)

                scheduler = sandbox.create_scheduler(
                    lab.CryptaLab,
                    schedule_interval=task.schedule_interval,
                    kill_timeout=task.kill_timeout,
                    extra_params={
                        "task_name": task.task,
                        "task_args": {"day": day},
                        "override_basic_params": True,
                        "overridden_semaphore": "{}_{}_{}".format(task.task, env, day),
                        "override_send_juggler_events": True,
                        "override_juggler_host": crypta_common_task.JUGGLER_HOSTS[env],
                        "override_juggler_service": juggler_service,
                    },
                    env=env,
                    retry_interval=task.retry_interval,
                )
                if env == environment.STABLE:
                    scheduler.check(task.crit_time, juggler_service=juggler_service).add_yt_dependencies(YtProxy.Group.offline)

    return juggler


def add_rule_estimator_api(base_juggler_check_generator):
    deploy_gen = base_juggler_check_generator.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage = deploy_gen.stage("crypta-lab-rule-estimator-api")
    deploy_unit = stage.multi_cluster_deploy_unit(
        pods_per_cluster={dc_consts.SAS: 2},
        cpu_ms=500,
        ram=DataSize(gb=1),
        storage=deploy.HDD(
            capacity=DataSize(gb=5),
            bandwidth_guarantee=DataSize(mb=10),
            bandwidth_limit=DataSize(mb=10),
        ),
        project_network=yp_consts.CRYPTANETS,
        network_bandwidth=DataSize(mb=1),
        id_=environment.PRODUCTION,
    ).add_endpoint_set(
        port=GRPC_PORT,
        id_="grpc",
    ).add_endpoint_set(
        port=HTTP_PORT,
        id_="http",
    ).add_balancer(
        BALANCER,
    ).deployment_strategy(
        max_unavailable=1,
    )

    box = deploy_unit.box(
        docker_image="crypta/crypta-lab-rule-estimator-api",
    ).docker_release_rule(
        environment.STABLE,
    ).add_init_command(
        deploy.Command(
            "/root/init.sh"
        ),
    ).literal_env(
        crypta_env.Production.crypta_environment
    ).literal_env(
        crypta_env.LiteralEnv("GRPC_PORT", str(GRPC_PORT))
    ).literal_env(
        crypta_env.LiteralEnv("HTTP_PORT", str(HTTP_PORT))
    ).secret_env(
        # TODO(cherenkov-p-a) add metrics
        crypta_env.solomon_token
    ).secret_env(
        crypta_env.Production.rule_estimator_tvm_secret
    ).secret_env(
        crypta_env.Production.yt_token
    )

    box.workload(
        start_cmd="/root/api.sh",
        id_="grpc",
    ).liveness_cmd_check(
        deploy.Command(
            "/root/grpcurl -plaintext localhost:{} NCrypta.NLab.NRuleEstimator.RuleEstimator.Ping".format(GRPC_PORT)
        ),
    )

    box.workload(
        start_cmd="/root/http_proxy.sh",
        id_="http",
    ).liveness_http_check(
        path="/ping",
        port=HTTP_PORT,
    )

    juggler = deploy_unit.get_juggler_generator(base_juggler_check_generator.clone(
        warn_limit=0,
        crit_limit=100,
        escalation=False,
        host="crypta-lab-rule-estimator-api",
    ))

    flap_detector_params = FlapDetectorParams(timedelta(minutes=5), timedelta(minutes=10))
    juggler.icmpping().add_flap_detector(flap_detector_params)
    juggler.http(
        "alive",
        path="/ping",
        port=HTTP_PORT,
    ).add_flap_detector(flap_detector_params)


def add_rule_estimator_worker(base_juggler_check_generator):
    deploy_gen = base_juggler_check_generator.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage = deploy_gen.stage("crypta-lab-rule-estimator-worker")
    deploy_unit = stage.multi_cluster_deploy_unit(
        pods_per_cluster={dc_consts.SAS: 2},
        cpu_ms=2000,
        ram=DataSize(gb=16),
        storage=deploy.HDD(
            capacity=DataSize(gb=5),
            bandwidth_guarantee=DataSize(mb=10),
            bandwidth_limit=DataSize(mb=10),
        ),
        project_network=yp_consts.CRYPTANETS,
        network_bandwidth=DataSize(mb=10),
        id_=environment.PRODUCTION,
    ).add_endpoint_set(
        port=SOLOMON_PORT,
        id_="solomon",
    ).deployment_strategy(
        max_unavailable=1,
    )

    box = deploy_unit.box(
        docker_image="crypta/crypta-lab-rule-estimator-worker",
    ).docker_release_rule(
        environment.STABLE,
    ).add_init_command(
        deploy.Command(
            "/root/init.sh"
        ),
    ).literal_env(
        crypta_env.Production.crypta_environment
    ).literal_env(
        crypta_env.LiteralEnv("RULE_ESTIMATOR_API_ENDPOINT", "{}:{}".format(BALANCER, GRPC_PORT))
    ).secret_env(
        # TODO(cherenkov-p-a) add metrics
        crypta_env.solomon_token
    ).secret_env(
        crypta_env.Production.rule_estimator_tvm_secret
    ).secret_env(
        crypta_env.Production.yt_token
    ).secret_env(
        crypta_env.crypta_api_token,
    ).secret_env(
        crypta_env.Production.yql_token
    )

    box.workload(
        start_cmd="/root/worker.sh",
        id_="worker",
    )

    juggler = deploy_unit.get_juggler_generator(base_juggler_check_generator.clone(
        warn_limit=0,
        crit_limit=100,
        escalation=False,
        host="crypta-lab-rule-estimator-worker",
    ))

    flap_detector_params = FlapDetectorParams(timedelta(minutes=5), timedelta(minutes=10))
    juggler.icmpping().add_flap_detector(flap_detector_params)

    solomon = solomon_check_generator.DcSolomonCheckGenerator(
        juggler,
        project="crypta_lab",
        service="stats",
        cluster="rule_estimator_worker",
        dc="sas",
    )
    for task_type in ("rule", "rule_condition"):
        solomon.get_sensor({"sensor": "errors", "task_type": task_type}).create_threshold_check(
            aggregation=alert_pb2.MAX,
            predicate=alert_pb2.GT,
            threshold=0,
            period=timedelta(hours=1),
            description="Errors: {{ pointValue }}",
        )

    solomon.create_logbroker_lag_check(
        consumer="crypta/prod/lab/rule-estimates-consumer",
        topic="crypta/prod/lab/rule-estimates-update-log",
        threshold=300,
        period=timedelta(minutes=5),
    )
