package binutils

import (
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"sync"
	"syscall"
	"time"
)

type Task struct {
	Run      func()
	Shutdown func()
}

func MakeHTTPTask(server *http.Server) Task {
	return Task{
		Run: func() {
			log.Printf("Server start %s", server.Addr)
			if err := server.ListenAndServe(); err != nil && err != http.ErrServerClosed {
				log.Fatalf("server failed: %s", err)
			}
		},
		Shutdown: func() {
			log.Printf("Server is stopping %s", server.Addr)
			ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
			defer cancel()

			if err := server.Shutdown(ctx); err != nil {
				log.Fatalf("Failed to shutdown server %s: %v", server.Addr, err)
			}
			log.Printf("Stopped %s", server.Addr)
		},
	}
}

func RunAllAndWaitForSignal(tasks ...Task) {
	var wg sync.WaitGroup
	wg.Add(len(tasks))

	for _, task := range tasks {
		task := task
		go func() {
			defer wg.Done()

			stop := make(chan os.Signal, 1)
			signal.Notify(stop, os.Interrupt, syscall.SIGINT, syscall.SIGTERM)

			finished := make(chan struct{})
			go func() {
				task.Run()
				finished <- struct{}{}
			}()

			select {
			case <-stop:
			case <-finished:
			}

			task.Shutdown()
		}()
	}
	wg.Wait()
}
