package httpproxy

import (
	"a.yandex-team.ru/crypta/lib/go/binutils"
	"fmt"
	"github.com/grpc-ecosystem/grpc-gateway/runtime"
	"github.com/spf13/pflag"
	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"gopkg.in/yaml.v2"
	"net/http"
	"os"
)

type RegisterHandleFunc func(context.Context, *runtime.ServeMux, string, []grpc.DialOption) error

type HTTPConfig struct {
	Port int64 `yaml:"port"`
}

type ServiceConfig struct {
	Host string `yaml:"host"`
	Port int64  `yaml:"port"`
}

type Config struct {
	HTTP    HTTPConfig    `yaml:"http"`
	Service ServiceConfig `yaml:"service"`
}

func GetConfig() (Config, error) {
	configPath := pflag.String("config", "", "Config file")
	pflag.Parse()

	configFile, err := os.Open(*configPath)
	if err != nil {
		return Config{}, err
	}
	defer configFile.Close()

	var config Config
	if err = yaml.NewDecoder(configFile).Decode(&config); err != nil {
		return Config{}, err
	}

	return config, nil
}

func CreateServer(config Config, registerHandleFromEndpointFunc RegisterHandleFunc) *http.Server {
	ctx := context.Background()
	mux := runtime.NewServeMux()
	serviceEndpoint := fmt.Sprintf("%s:%d", config.Service.Host, config.Service.Port)
	opts := []grpc.DialOption{grpc.WithInsecure()}

	err := registerHandleFromEndpointFunc(ctx, mux, serviceEndpoint, opts)
	if err != nil {
		panic(err)
	}

	port := fmt.Sprintf(":%d", config.HTTP.Port)
	srv := &http.Server{Addr: port, Handler: mux}

	return srv
}

func RunHTTPProxy(registerHandleFromEndpointFunc RegisterHandleFunc) {
	config, err := GetConfig()
	if err != nil {
		panic(err)
	}

	httpProxy := CreateServer(config, registerHandleFromEndpointFunc)

	binutils.RunAllAndWaitForSignal(
		binutils.MakeHTTPTask(httpProxy),
	)
}
