package kv

import (
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"context"
	"fmt"
)

type keyValueRecord struct {
	Key   string `yson:"key,key"`
	Value string `yson:"value"`
}

type keyRecord struct {
	Key string `yson:"key,key"`
}

type ReadOnlyClient interface {
	Lookup(ctx context.Context, key string) (*string, error)
	LookupMany(ctx context.Context, keys []string) (map[string]string, error)

	Select(ctx context.Context, query string) (map[string]string, error)
}

type clientBase interface {
	ReadOnlyClient

	Write(ctx context.Context, key string, value string) error
	Delete(ctx context.Context, key string) error

	WriteMany(ctx context.Context, values map[string]string) error
	DeleteMany(ctx context.Context, keys []string) error
}

type Tx interface {
	clientBase

	ID() yt.TxID
	Abort() error
	Commit() error
}

type Client interface {
	clientBase
	StartTx(ctx context.Context) (Tx, error)
}

type clientBaseImpl struct {
	yt   yt.TabletClient
	path ypath.Path
}

type txImpl struct {
	clientBaseImpl
	tx yt.TabletTx
}

type clientImpl struct {
	clientBaseImpl
	yt yt.Client
}

func (client *clientBaseImpl) Lookup(ctx context.Context, key string) (*string, error) {
	records, err := client.LookupMany(ctx, []string{key})
	if err != nil {
		return nil, err
	}

	var value *string
	if _, ok := records[key]; ok {
		value = new(string)
		*value = records[key]

	}
	return value, nil
}

func (client *clientBaseImpl) Write(ctx context.Context, key string, value string) error {
	return client.WriteMany(ctx, map[string]string{key: value})
}

func (client *clientBaseImpl) Delete(ctx context.Context, key string) error {
	return client.DeleteMany(ctx, []string{key})
}

func (client *clientBaseImpl) LookupMany(ctx context.Context, keys []string) (map[string]string, error) {
	tableReader, err := client.yt.LookupRows(ctx, client.path, serializeKeys(keys), nil)
	if err != nil {
		return nil, err
	}

	return deserializeRecords(tableReader)
}

func (client *clientBaseImpl) WriteMany(ctx context.Context, records map[string]string) error {
	return client.yt.InsertRows(ctx, client.path, serializeRecords(records), nil)
}

func (client *clientBaseImpl) DeleteMany(ctx context.Context, keys []string) error {
	return client.yt.DeleteRows(ctx, ypath.Path(client.path), serializeKeys(keys), nil)
}

func (client *clientBaseImpl) Select(ctx context.Context, query string) (map[string]string, error) {
	fullQuery := fmt.Sprintf("* FROM [%s] %s", client.path, query)

	tableReader, err := client.yt.SelectRows(ctx, fullQuery, nil)
	if err != nil {
		return nil, err
	}

	return deserializeRecords(tableReader)
}

func (client *clientImpl) StartTx(ctx context.Context) (Tx, error) {
	tx, err := client.yt.BeginTabletTx(ctx, nil)
	if err != nil {
		return nil, err
	}

	return &txImpl{
		clientBaseImpl: clientBaseImpl{
			yt:   tx,
			path: client.path,
		},
		tx: tx,
	}, nil
}

func (tx *txImpl) ID() yt.TxID {
	return tx.tx.ID()
}

func (tx *txImpl) Abort() error {
	return tx.tx.Abort()
}

func (tx *txImpl) Commit() error {
	return tx.tx.Commit()
}

func NewClient(yt yt.Client, path ypath.Path) Client {
	return &clientImpl{
		clientBaseImpl: clientBaseImpl{
			yt:   yt,
			path: path,
		},
		yt: yt,
	}
}

func serializeKeys(keys []string) []interface{} {
	result := make([]interface{}, len(keys))

	for i := range keys {
		result[i] = keyRecord{keys[i]}
	}
	return result
}

func deserializeRecords(tableReader yt.TableReader) (map[string]string, error) {
	defer tableReader.Close()

	result := make(map[string]string)
	for tableReader.Next() {
		var record keyValueRecord
		err := tableReader.Scan(&record)
		if err != nil {
			return nil, err
		}
		result[record.Key] = record.Value
	}
	return result, nil
}

func serializeRecords(records map[string]string) []interface{} {
	var result []interface{}

	for key, value := range records {
		result = append(result, &keyValueRecord{key, value})
	}
	return result
}
