package ru.yandex.crypta.clients.audience;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;
import javax.inject.Named;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import okhttp3.HttpUrl;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import org.json.JSONObject;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.jackson.JacksonConverterFactory;

import ru.yandex.crypta.clients.tvm.AudienceTvmClient;
import ru.yandex.crypta.clients.tvm.TvmClient;
import ru.yandex.crypta.clients.tvm.TvmOkHttpInterceptor;
import ru.yandex.crypta.clients.utils.OkHttpUtils;
import ru.yandex.crypta.clients.utils.RestClientWithOauth;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lab.proto.TAudienceTvmConfig;
import ru.yandex.crypta.lib.proto.TAudienceApiConfig;

public class DefaultAudienceClient extends RestClientWithOauth implements AudienceClient {
    private final TAudienceApiConfig config;
    private final AudienceTvmClient unicornTvm;
    private final AudienceTvmClient cryptaidorTvm;
    private final TAudienceTvmConfig audienceTvmConfig;

    @Inject
    public DefaultAudienceClient(
            TAudienceApiConfig config,
            TvmClient tvm,
            @Named("unicornTvm") AudienceTvmClient unicornTvm,
            @Named("cryptaidorTvm") AudienceTvmClient cryptaidorTvm,
            TAudienceTvmConfig audienceTvmConfig
    ) {
        super(config.getToken());
        this.config = config;
        this.unicornTvm = unicornTvm;
        this.cryptaidorTvm = cryptaidorTvm;
        this.audienceTvmConfig = audienceTvmConfig;

        List<Integer> dstIds = List.of(audienceTvmConfig.getDestinationTvmId());
        tvm.setDstClientIds(dstIds);
        unicornTvm.setDstClientIds(dstIds);
        cryptaidorTvm.setDstClientIds(dstIds);
    }

    private String parseErrorMessage(ResponseBody responseBody) {
        String message = "";

        if (responseBody != null) {
            try {
                JSONObject obj = new JSONObject(responseBody.string());
                message = obj.getString("message");
            } catch (IOException e) {
                throw Exceptions.unsupported();
            }
        }

        return message;
    }

    private <T> Response<T> tryResponse(Response<T> response) {
        Integer code = response.code();

        if (Objects.equals(code, OkHttpUtils.NOT_FOUND)) {
            throw Exceptions.notFound();
        }

        if (Objects.equals(code, OkHttpUtils.FORBIDDEN_STATUS)) {
            throw Exceptions.forbidden("Forbidden", String.valueOf(response.code()));
        }

        if (Objects.equals(code, 429)) {
            throw Exceptions.tooManyRequestsException();
        }

        if (code >= 400) {
            String message = parseErrorMessage(response.errorBody());
            throw Exceptions.wrongRequestException(message, "AUDIENCE_API_RESPONSE");
        }

        return response;
    }

    private Map<Integer, JsonNode> fetchSegments() {
        try {
            var response = tryResponse(getOauthAudienceApi().listSegments().execute());

            if (response.body() != null) {
                Map<Integer, JsonNode> segmentsMap = new HashMap<>();
                response.body().withArray("segments").forEach(segment -> {
                    segmentsMap.put(segment.get("id").asInt(), segment);
                });

                return segmentsMap;
            }

            return null;
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public JsonNode listSegments() {
        Map<Integer, JsonNode> segments = fetchSegments();
        ObjectMapper mapper = new ObjectMapper();

        return mapper.valueToTree(segments);
    }

    @Override
    public JsonNode getSegment(String segmentId) {
        return fetchSegments().getOrDefault(Integer.parseInt(segmentId), JsonNodeFactory.instance.nullNode());
    }

    @Override
    public AudienceListGrantsResponse listGrants(String segmentId) {
        try {
            var response = tryResponse(getOauthAudienceApi().getGrants(segmentId).execute());

            return response.body();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public AudienceCreateGrantResponse createGrant(String segmentId, String userLogin, String comment) {
        try {
            var userObj = new JSONObject();
            var grantObj = new JSONObject();
            userObj.put("user_login", userLogin);
            userObj.put("comment", comment);
            grantObj.put("grant", userObj);

            var requestBody = RequestBody.create(
                    okhttp3.MediaType.parse("application/json; charset=utf-8"),
                    grantObj.toString());
            var response = tryResponse(
                    getOauthAudienceApi().createGrant(segmentId, requestBody).execute()
            );

            return response.body();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public AudienceDeleteResponse deleteGrant(String segmentId, String userLogin) {
        try {
            var response = tryResponse(getOauthAudienceApi().deleteGrant(segmentId, userLogin).execute());

            return response.body();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }


    @Override
    public AudienceDeleteResponse deleteSegment(String segmentId) {
        try {
            var response = tryResponse(getOauthAudienceApi().deleteSegment(segmentId).execute());
            return response.body();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public JsonNode updateSegment(String segmentId, String name) {
        try {
            var segmentObj = new JSONObject();
            var nameObj = new JSONObject();
            nameObj.put("name", name);
            segmentObj.put("segment", nameObj);

            var requestBody = RequestBody.create(
                    okhttp3.MediaType.parse("application/json; charset=utf-8"),
                    segmentObj.toString());
            var response = tryResponse(getOauthAudienceApi().updateSegment(segmentId, requestBody).execute());

            return response.body();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public JsonNode uploadSegmentFile(String login, boolean exportCryptaId) {
        AudienceApi audienceApi = getAudienceApi(exportCryptaId);

        try {
            var response = tryResponse(
                    audienceApi.uploadSegmentFile(
                            login,
                            MultipartBody.Part.createFormData(
                                    "file",
                                    "data.tsv",
                                    RequestBody.create(okhttp3.MediaType.parse("multipart/form-data"), new byte[0])))
                            .execute()
            );

            return response.body();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public JsonNode confirmSegment(int segmentId, String segmentName, String login, boolean exportCryptaId) {
        AudienceApi audienceApi = getAudienceApi(exportCryptaId);

        try {
            var segmentObj = new JSONObject();
            var infoObj = new JSONObject();

            infoObj.put("id", segmentId);
            infoObj.put("name", segmentName);
            infoObj.put("hashed", 0);
            infoObj.put("content_type", "yuid");

            segmentObj.put("segment", infoObj);

            var requestBody = RequestBody.create(
                    okhttp3.MediaType.parse("application/json; charset=utf-8"),
                    segmentObj.toString());
            var response = tryResponse(
                    audienceApi.confirmSegment(segmentId, requestBody, login, false
                    ).execute());

            return response.body();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    private AudienceApi getAudienceApi(boolean exportCryptaId) {
        return exportCryptaId ? getCryptaidorAudienceApi() : getPrivateAudienceApi();
    }

    private AudienceApi getAudienceTvmClient(TvmClient tvmClient, int dstId) {
        var client = new OkHttpClient.Builder()
                .addInterceptor(new TvmOkHttpInterceptor(tvmClient, dstId))
                .build();

        return createAudienceApi(client);
    }

    private AudienceApi getOauthAudienceApi() {
        var client = new OkHttpClient.Builder()
                .addInterceptor(chain -> {
                    var request = chain.request().newBuilder()
                            .header("Authorization", "OAuth " + config.getToken())
                            .build();
                    return chain.proceed(request);
                })
                .readTimeout(10, TimeUnit.SECONDS)
                .build();

        return createAudienceApi(client);
    }

    private AudienceApi createAudienceApi(OkHttpClient client) {
        HttpUrl url = new HttpUrl.Builder().scheme(config.getScheme()).host(config.getUrl()).port(config.getPort()).build();

        var retrofit =  new Retrofit.Builder()
                .baseUrl(url)
                .addConverterFactory(JacksonConverterFactory.create())
                .client(client)
                .build();

        return retrofit.create(AudienceApi.class);
    }

    private AudienceApi getPrivateAudienceApi() {
        return getAudienceTvmClient(unicornTvm, audienceTvmConfig.getDestinationTvmId());
    }

    private AudienceApi getCryptaidorAudienceApi() {
        return getAudienceTvmClient(cryptaidorTvm, audienceTvmConfig.getDestinationTvmId());
    }
}
