package ru.yandex.crypta.clients.bigb;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import yabs.proto.Profile;

import ru.yandex.ads.bsyeti.libs.counter_lib.proto.TCounterPack;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.crypta.clients.tvm.TvmClient;
import ru.yandex.crypta.clients.tvm.TvmOkHttpInterceptor;
import ru.yandex.crypta.clients.utils.OkHttpUtils;
import ru.yandex.crypta.clients.utils.OnlyInet6AddressDns;
import ru.yandex.crypta.lib.proto.TBigbTvmConfig;
import ru.yandex.crypta.lib.proto.TBigbViewerTvmConfig;
import ru.yandex.yabs.server.proto.keywords.EKeyword;

import static ru.yandex.crypta.clients.utils.HttpExceptions.checkResponse;

public class DefaultBigbClient implements BigbClient {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultBigbClient.class);

    public static final TypeReference<Map<String, Object>> MAP_TYPE_REFERENCE =
            new TypeReference<Map<String, Object>>() {
            };
    private static final String BIGB_FAST_URL = "http://bigb-fast.yandex.ru/";
    private static final String BIGB_PROFILE_VIEWER_HOST = "bb-viewer.yandex-team.ru";

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private final OkHttpClient bigbClient;
    private final String bigbClientName;

    private final OkHttpClient bigbViewerClient;

    @Inject
    public DefaultBigbClient(TBigbTvmConfig bigbTvm, TBigbViewerTvmConfig bigbViewerTvm, TvmClient tvm) {
        tvm.setDstClientIds(Cf.list(bigbTvm.getDestinationTvmId(), bigbViewerTvm.getDestinationTvmId()));
        this.bigbClientName = bigbTvm.getClientName();
        this.bigbClient = new OkHttpClient.Builder()
                .dns(new OnlyInet6AddressDns())
                .readTimeout(5, TimeUnit.SECONDS)
                .addInterceptor(new TvmOkHttpInterceptor(tvm, bigbTvm.getDestinationTvmId()))
                .build();
        this.bigbViewerClient = new OkHttpClient.Builder()
                .dns(new OnlyInet6AddressDns())
                .readTimeout(60, TimeUnit.SECONDS)
                .addInterceptor(new TvmOkHttpInterceptor(tvm, bigbViewerTvm.getDestinationTvmId()))
                .build();
    }


    @Override
    public JsonNode getBigbData(BigbIdType bigbIdType, String bigbIdValue) {
        Request request = getBigbRequestFor(bigbIdType, bigbIdValue, false);
        return getBigBResponseJsonNode(request);
    }

    @Override
    public Profile getBigbDataProto(BigbIdType idType, String id) {
        Request request = getBigbRequestFor(idType, id, true);
        return getBigbResponseProto(request);
    }

    @Override
    public JsonNode getCommonProfile(CommonId commonId) {
        Request request = getBigbRequestFor(commonId, false);
        return stringifyCommonProfile(getBigBResponseJsonNode(request));
    }

    /**
     * Replaces integer values of common_profile_id and query_id fields of BigB common profile json
     * with strings to have them passed to JavaScript code of the viewers properly.
     *
     * @param commonProfile the whole BigB common profile.
     * @return JsonNode BigB common profile with stringified values of common_profile_id and query_id fields.
     */
    private JsonNode stringifyCommonProfile(JsonNode commonProfile) {
        ObjectNode commonProfileObject = (ObjectNode) commonProfile;

        if (commonProfile.has(Fields.COMMON_PROFILE_ID)) {
            commonProfileObject.set(
                    Fields.COMMON_PROFILE_ID,
                    OBJECT_MAPPER
                            .convertValue(commonProfile.get(Fields.COMMON_PROFILE_ID).asText(), JsonNode.class)
            );
        }

        if (commonProfile.has(Fields.COMMON_PROFILE_QUERIES)) {
            JsonNode queriesNode = commonProfile.get(Fields.COMMON_PROFILE_QUERIES);
            for (int i = 0; i < queriesNode.size(); i++) {
                commonProfileObject.withArray(Fields.COMMON_PROFILE_QUERIES).set(
                        i,
                        ((ObjectNode) queriesNode.get(i)).set(
                                Fields.COMMON_PROFILE_QUERY_ID,
                                OBJECT_MAPPER
                                        .convertValue(
                                                queriesNode.get(i).get(Fields.COMMON_PROFILE_QUERY_ID).asText(),
                                                JsonNode.class)
                        )
                );
            }
        }
        return OBJECT_MAPPER.convertValue(commonProfileObject, JsonNode.class);
    }

    private JsonNode getBigBResponseJsonNode(Request request) {
        try (Response response = bigbClient.newCall(request).execute()) {
            checkResponse(response);
            return OkHttpUtils.getResponseJson(response);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Profile getBigbResponseProto(Request request) {
        try (Response response = bigbClient.newCall(request).execute()){
            checkResponse(response);
            return OkHttpUtils.getResponseProto(response, Profile.parser());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private JsonNode decodeHouseholds(String householdEncoded) {
        return OBJECT_MAPPER.convertValue(Household.from(householdEncoded), JsonNode.class);
    }

    private JsonNode replaceHouseholdWithDecrypted(JsonNode jsonProfile) {
        if (jsonProfile.has(Fields.HOUSEHOLD)) {
            // There is only one node of households by agreement
            JsonNode household = jsonProfile.get(Fields.HOUSEHOLD).get(0);
            ObjectNode profileObject = OBJECT_MAPPER.convertValue(jsonProfile, ObjectNode.class);

            try {
                ((ArrayNode) profileObject.get(Fields.HOUSEHOLD))
                        .set(0, OBJECT_MAPPER
                                .convertValue(
                                        ((ObjectNode) household).set(
                                                Fields.VALUE,
                                                decodeHouseholds(household.get(Fields.VALUE).asText())
                                        ),
                                        JsonNode.class
                                ));
            } catch (RuntimeException e) {
                LOG.debug("Unable to decode households, left encoded: " + household.get(Fields.VALUE).asText());
            }

            return OBJECT_MAPPER.convertValue(profileObject, JsonNode.class);
        }

        return jsonProfile;
    }

    @Override
    public JsonNode getEnhancedProfile(BigbIdType uidType, String uidValue, int matching, String cookie) {
        Request request = getEnhancedProfileRequest(uidType, uidValue, matching, cookie);
        try (Response response = bigbViewerClient.newCall(request).execute()) {
            checkResponse(response);
            return replaceHouseholdWithDecrypted(OkHttpUtils.getResponseJson(response));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public JsonNode getUserCryptaHubs(BigbIdType uidType, String uidValue, String cookie) {
        Set<String> hubIdSet = new HashSet<>();
        JsonNode profile = getEnhancedProfile(uidType, uidValue, 1, cookie);

        if (profile.has(Fields.CRYPTA_HUB_ID)) {
            JsonNode hubs = profile.get(Fields.CRYPTA_HUB_ID);
            hubs.forEach(hub -> hub.get("users")
                    .forEach(user -> hubIdSet.add(user.get(Fields.UNIQ_ID).asText()))
            );
        }

        return OBJECT_MAPPER.convertValue(hubIdSet, JsonNode.class);
    }

    @Override
    public List<TCounterPack> getCounters(BigbIdType idType, String idValue, Collection<Integer> counters) {
        Request request = getBigCountersRequest(idType, idValue, counters);
        return getBigbResponseProto(request).getPackedCountersList();
    }

    @Override
    public JsonNode getUserSearchTextWithMatching(BigbIdType uidType, String uidValue, String cookie) {
        Map<String, Object> searchText = new HashMap<>();

        JsonNode uids = getUserCryptaHubs(uidType, uidValue, cookie);
        ArrayList<String> uidsList = OBJECT_MAPPER.convertValue(uids, new TypeReference<ArrayList<String>>() {
        });

        if (uids.size() == 0) {
            uidsList.add(uidValue);
        }

        uidsList.forEach(uid -> {
            JsonNode userSearchText = getUserSearchText(uidType, uidValue, 0, cookie);
            searchText.putAll(OBJECT_MAPPER.convertValue(userSearchText, MAP_TYPE_REFERENCE));
        });

        return OBJECT_MAPPER.convertValue(searchText, JsonNode.class);
    }

    @Override
    public JsonNode getUserSearchText(BigbIdType uidType, String uidValue, int matching, String cookie) {
        Request request = getEnhancedProfileRequest(uidType, uidValue, matching, cookie);
        try (Response response = bigbViewerClient.newCall(request).execute()){
            checkResponse(response);

            JsonNode jsonNode = OBJECT_MAPPER.readTree(Objects.requireNonNull(response.body()).string());
            Map<String, Object> searchTextByQueryId = new HashMap<>();

            if (jsonNode.has(Fields.PROFILE_SEARCH_TEXT)) {
                jsonNode.get(Fields.PROFILE_SEARCH_TEXT).forEach(item -> {
                    Object newItem = ((ObjectNode) item).put(Fields.UNIQ_ID, uidValue);
                    searchTextByQueryId
                            .put(item.get(Fields.QUERY_ID).asText().replaceFirst("^0+(?!$)", ""), newItem);
                });
            }

            return OBJECT_MAPPER.convertValue(searchTextByQueryId, JsonNode.class);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Request getEnhancedProfileRequest(BigbIdType uidType, String uidValue, int matching, String cookie) {
        HttpUrl.Builder urlBuilder = new HttpUrl.Builder()
                .scheme("https")
                .host(BIGB_PROFILE_VIEWER_HOST)
                .addPathSegment("profile")
                .addQueryParameter(uidType.value(), uidValue)
                .addQueryParameter("glue", Integer.toString(matching));

        if (uidType.value().equals(BigbIdType.DUID.value())) {
            urlBuilder.addQueryParameter(BigbIdType.BIGB_UID.value(), uidValue);
        }

        HttpUrl url = urlBuilder.build();
        return new Request.Builder()
                .url(url)
                .addHeader(Headers.COOKIE, cookie)
                .get()
                .build();
    }

    private Request getBigbRequestFor(BigbIdType bigbIdType, String bigbIdValue, boolean isProto) {
        HttpUrl.Builder url = HttpUrl
                .parse(BIGB_FAST_URL)
                .newBuilder()
                .addPathSegment("bigb")
                .addQueryParameter(bigbIdType.value(), bigbIdValue)
                .addQueryParameter("client", bigbClientName);

        if (isProto) {
            url.addQueryParameter("format", "protobuf");
        }

        if (bigbIdType.value().equals(BigbIdType.DUID.value())) {
            url.addQueryParameter(BigbIdType.BIGB_UID.value(), bigbIdValue);
        }

        return new Request.Builder()
                .url(url.toString())
                .get()
                .build();
    }

    private Request getBigCountersRequest(BigbIdType bigbIdType, String bigbIdValue, Collection<Integer> counters) {
        HttpUrl.Builder url = HttpUrl
                .parse(BIGB_FAST_URL)
                .newBuilder()
                .addPathSegment("bigb")
                .addQueryParameter(bigbIdType.value(), bigbIdValue)
                .addQueryParameter("client", bigbClientName)
                .addQueryParameter("format", "protobuf")
                .addQueryParameter("counters", counters.stream().map(String::valueOf).collect(Collectors.joining(",")))
                .addQueryParameter("keywords", String.valueOf(EKeyword.KW_BT_COUNTER_VALUE));

        if (bigbIdType.value().equals(BigbIdType.DUID.value())) {
            url.addQueryParameter(BigbIdType.BIGB_UID.value(), bigbIdValue);
        }

        return new Request.Builder()
                .url(url.toString())
                .get()
                .build();
    }

    private Request getBigbRequestFor(CommonId commonProfileId, boolean isProto) {
        HttpUrl.Builder url = HttpUrl
                .parse(BIGB_FAST_URL)
                .newBuilder()
                .addPathSegment("common_profiles")
                .addQueryParameter("common_profile_id", commonProfileId.getValue());

        if (isProto) {
            url.addQueryParameter("format", "protobuf");
        }

        return new Request.Builder()
                .url(url.toString())
                .get()
                .build();
    }
}
