package ru.yandex.crypta.clients.blackbox;

import java.io.IOException;
import java.net.InetAddress;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.crypta.clients.tvm.TvmClient;
import ru.yandex.crypta.clients.tvm.TvmOkHttpInterceptor;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.proto.TBlackboxConfig;

public class HttpBlackboxClient implements BlackboxClient {

    private static final Logger LOG = LoggerFactory.getLogger(HttpBlackboxClient.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final String HTTP_SCHEME = "http";
    private static final String BLACKBOX_URL_PATH = "blackbox";
    private static final String AUTH_STATUS_VALID = "VALID";
    private static final String SESSION_ID_STATUS_NEED_RESET = "NEED_RESET";

    private final String blackboxUrl;
    private final OkHttpClient httpClient;
    private final Cache<String, AuthResult> oauthTokenCache = CacheBuilder.newBuilder()
            .maximumSize(1000)
            .expireAfterWrite(2, TimeUnit.MINUTES)
            .build();
    private final Cache<String, AuthResult> sessionCache = CacheBuilder.newBuilder()
            .maximumSize(1000)
            .expireAfterAccess(2, TimeUnit.MINUTES)
            .build();

    @Inject
    public HttpBlackboxClient(TBlackboxConfig blackbox, TvmClient tvm) {
        this.blackboxUrl = blackbox.getUrl();
        tvm.setDstClientIds(Cf.list(blackbox.getDstTvmId()));
        this.httpClient = new OkHttpClient.Builder()
                .addInterceptor(new TvmOkHttpInterceptor(tvm, blackbox.getDstTvmId()))
                .build();
    }

    @Override
    public AuthResult oauth(String oauthHeader, InetAddress userip) {
        return Caching.fetch(oauthTokenCache, oauthHeader, () -> getOauthUserInfo(oauthHeader, userip));
    }

    @Override
    public AuthResult sessionid(String sessionid, InetAddress userip, String host) {
        return Caching.fetch(sessionCache, sessionid, () -> getSessionUserInfo(sessionid, userip, host));
    }

    public AuthResult getSessionUserInfo(String sessionid, InetAddress userip, String host) {
        Request request = getSessionRequest(sessionid,
                userip.getHostAddress(),
                host
        );
        try (Response response = httpClient.newCall(request).execute()){
            return getAuthResult(AuthResult.Method.SESSIONID, getJsonBody(response));
        } catch (IOException e) {
            throw Exceptions.unavailable();
        }
    }

    private AuthResult getOauthUserInfo(String oauthHeader, InetAddress userip) {
        Request request = getOAuthRequest(oauthHeader, userip.getHostAddress());
        try (Response response = httpClient.newCall(request).execute()){
            return getAuthResult(AuthResult.Method.OAUTH, getJsonBody(response));
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    private AuthResult getAuthResult(AuthResult.Method method, JsonNode jsonResponse) {

        if (!isStatusValid(jsonResponse)) {
            if (isStatusNeedReset(jsonResponse)) {
                return AuthResult.needReset(method);
            }

            return AuthResult.fail(method);
        }

        var login = getLogin(jsonResponse);
        if (!login.isPresent()) {
            return AuthResult.fail(method);
        }
        var puid = getPuid(jsonResponse);

        return AuthResult.ok(method, login.get(), puid.orElse(""));
    }

    private boolean hasStatusValue(JsonNode response) {
        JsonNode status = response.get("status");
        if (status == null) {
            return false;
        }
        JsonNode statusValue = status.get("value");

        return statusValue != null;
    }

    private boolean responseHasStatusValue(JsonNode response, String statusValue) {
        if (hasStatusValue(response)) {
            return response.get("status").get("value").asText().equals(statusValue);
        }

        return false;
    }

    private boolean isStatusValid(JsonNode response) {
        return responseHasStatusValue(response, AUTH_STATUS_VALID);
    }

    private boolean isStatusNeedReset(JsonNode response) {
        return responseHasStatusValue(response, SESSION_ID_STATUS_NEED_RESET);
    }

    private Optional<String> getLogin(JsonNode response) {
        return Optional.ofNullable(response.get("login")).map(JsonNode::asText);
    }

    private Optional<String> getPuid(JsonNode response) {
        return Optional.ofNullable(response.get("uid")).map(x -> x.get("value")).map(JsonNode::asText);
    }

    private Request getOAuthRequest(String oauthToken, String userip) {
        HttpUrl url = new HttpUrl.Builder()
                .scheme(HTTP_SCHEME)
                .host(blackboxUrl)
                .addPathSegment(BLACKBOX_URL_PATH)
                .addQueryParameter("method", "oauth")
                .addQueryParameter("userip", userip)
                .addQueryParameter("format", "json")
                .addQueryParameter("scopes", "crypta:api")
                .build();
        return new Request.Builder()
                .url(url)
                .addHeader("authorization", oauthToken)
                .get()
                .build();
    }

    private Request getSessionRequest(String sessionid, String userip, String host) {
        HttpUrl url = new HttpUrl.Builder()
                .scheme(HTTP_SCHEME)
                .host(blackboxUrl)
                .addPathSegment(BLACKBOX_URL_PATH)
                .addQueryParameter("method", "sessionid")
                .addQueryParameter("sessionid", sessionid)
                .addQueryParameter("userip", userip)
                .addQueryParameter("host", host)
                .addQueryParameter("format", "json")
                .build();
        return new Request.Builder()
                .url(url)
                .get()
                .build();
    }

    private JsonNode getJsonBody(Response response) {
        try {
            String responseBody = response.body().string();
            LOG.info("BlackBox reply: {}", responseBody);
            return OBJECT_MAPPER.readValue(responseBody, JsonNode.class);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }
}
