package ru.yandex.crypta.clients.graphite;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.charset.Charset;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.proto.TGraphiteConfig;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.lang.Validate;

public class DefaultGraphiteClient implements GraphiteClient {

    public static final int TIMEOUT = 500;
    public static final int CAPACITY = 100000;
    public static final int MAX_ATTEMPTS = 1000;
    public static final Duration TASK_TIMEOUT = Duration.ofMinutes(10);

    private static class ReportMetricTask {
        private final HostPort graphite;
        private final FullName fullName;
        private final Timestamp timestamp;
        private final Value value;
        private final Instant started;
        private int attempt;

        private ReportMetricTask(HostPort graphite, FullName fullName, Timestamp timestamp, Value value, Instant started) {
            this.graphite = graphite;
            this.fullName = fullName;
            this.timestamp = timestamp;
            this.value = value;
            this.started = started;
            this.attempt = 0;
        }

        public HostPort getGraphite() {
            return graphite;
        }

        public FullName getFullName() {
            return fullName;
        }

        public Timestamp getTimestamp() {
            return timestamp;
        }

        public Value getValue() {
            return value;
        }

        public Instant getStarted() {
            return started;
        }

        public int getAttempt() {
            return attempt;
        }

        public ReportMetricTask anotherAttempt() {
            ReportMetricTask task = new ReportMetricTask(getGraphite(), getFullName(), getTimestamp(), getValue(), getStarted());
            task.attempt = attempt + 1;
            return task;
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(DefaultGraphiteClient.class);
    private final ScheduledExecutorService executor;
    private final Map<HostPort, BlockingDeque<ReportMetricTask>> graphiteDeques;

    @Inject
    @SuppressWarnings("FutureReturnValueIgnored")
    public DefaultGraphiteClient(TGraphiteConfig graphite) {
        Validate.notEmpty(graphite.getHosts());

        this.executor = Executors.newSingleThreadScheduledExecutor();
        this.graphiteDeques = parseGraphiteHosts(graphite.getHosts()).stream().collect(Collectors.toMap(
                each -> each,
                each -> new LinkedBlockingDeque<ReportMetricTask>(CAPACITY)
        ));
        this.graphiteDeques.values().forEach(deque -> this.executor.scheduleWithFixedDelay(
                () -> sendMetric(deque), 0, 1, TimeUnit.NANOSECONDS)
        );
    }

    private List<HostPort> parseGraphiteHosts(String graphites) {
        return Stream.of(graphites.split(",")).map(HostPort::parse).collect(Collectors.toList());
    }

    private String formatMetric(ReportMetricTask task) {
        FullName fullName = task.getFullName();
        return String.format("%s.%s.%s.%s %f %d",
                fullName.getFrequency().getAlias(),
                fullName.getHostname(),
                fullName.getGroup(),
                fullName.getName(),
                task.getValue().getValue(),
                task.getTimestamp().getValue()
        );
    }

    private void withSocket(HostPort hostPort, Consumer<Socket> callback) {
        try (Socket socket = new Socket()) {
            InetSocketAddress endpoint = new InetSocketAddress(hostPort.getHost().format(), hostPort.getPort());
            socket.connect(endpoint, TIMEOUT);
            callback.accept(socket);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    private void write(Socket socket, String message) {
        try (Writer writer = new OutputStreamWriter(socket.getOutputStream(), Charset.defaultCharset())) {
            writer.write(message);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    private void sendMetric(BlockingDeque<ReportMetricTask> deque) {
        ReportMetricTask task;
        try {
            task = deque.pollFirst(100, TimeUnit.MILLISECONDS);
            if (task == null) {
                return;
            }
        } catch (InterruptedException e) {
            return;
        }
        String message = formatMetric(task);
        try {
            withSocket(task.getGraphite(), socket -> write(socket, message));
        } catch (RuntimeException e) {
            ReportMetricTask anotherAttempt = task.anotherAttempt();
            if (anotherAttempt.getAttempt() >= MAX_ATTEMPTS) {
                LOG.error("Graphite metric is lost on {} after {} attempts: {}", task.getGraphite(), MAX_ATTEMPTS, message);
                return;
            }
            Duration howLong = Duration.between(anotherAttempt.getStarted(), Instant.now());
            if (howLong.compareTo(TASK_TIMEOUT) >= 0) {
                LOG.error("Graphite metric is lost on {} due to timeout ({}): {}", task.getGraphite(), TASK_TIMEOUT, message);
                return;
            }
            if (!deque.offerLast(anotherAttempt)) {
                LOG.error("Graphite metric is lost on {} due to capacity limit: {}", task.getGraphite(), message);
            }
        }
        LOG.debug("Graphite metric reported to {} (attempt #{}): {}", task.getGraphite(), task.getAttempt(), message);
    }

    @Override
    public GraphiteResponse report(FullName fullName, Timestamp timestamp, Value value) {
        Instant now = Instant.now();
        long enqueuedCount = graphiteDeques.entrySet().stream().filter(entry -> {
            HostPort graphite = entry.getKey();
            BlockingDeque<ReportMetricTask> deque = entry.getValue();
            ReportMetricTask task = new ReportMetricTask(graphite, fullName, timestamp, value, now);
            return deque.offerLast(task);
        }).count();
        if (enqueuedCount == 0) {
            LOG.error(
                    "Graphite metric is lost due to capacity limit: {}",
                    formatMetric(new ReportMetricTask(null, fullName, timestamp, value, now))
            );
            throw Exceptions.unavailable();
        }
        return new GraphiteResponse(fullName, timestamp, value);
    }
}
