package ru.yandex.crypta.clients.idm;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.impl.DSL;

import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.data.EnhancedDSLContext;
import ru.yandex.crypta.idm.Role;
import ru.yandex.crypta.idm.Roles;

public class JooqRoleService implements RoleService {

    private final EnhancedDSLContext dsl;

    private Cache<String, List<Role>> rolesCache =
            CacheBuilder.newBuilder()
                    .expireAfterWrite(1, TimeUnit.MINUTES)
                    .build();

    @Inject
    public JooqRoleService(PostgresClient postgresClient) {
        this.dsl = new EnhancedDSLContext(postgresClient.getJooqConfiguration());
    }

    @Override
    public void addRole(String login, Role role) throws NoSuchRoleException {
        dsl.transaction(configuration -> {
            EnhancedDSLContext transactionDsl = new EnhancedDSLContext(configuration);

            transactionDsl.insertInto(Logins.TABLE)
                    .set(Logins.LOGIN, login)
                    .onDuplicateKeyIgnore()
                    .execute();

            boolean roleAvailable = Roles.INSTANCE.hasRole(role);
            if (!roleAvailable) {
                throw new NoSuchRoleException();
            }

            transactionDsl.insertInto(LoginsRoles.TABLE)
                    .set(LoginsRoles.LOGIN, login)
                    .set(LoginsRoles.ROLE, Roles.INSTANCE.toId(role))
                    .onDuplicateKeyIgnore()
                    .execute();
        });
    }

    @Override
    public void removeRole(String login, Role role) {
        String id = Roles.INSTANCE.toId(role);
        if (Objects.nonNull(id)) {
            dsl.deleteFrom(LoginsRoles.TABLE)
                    .where(LoginsRoles.LOGIN.eq(login))
                    .and(LoginsRoles.ROLE.eq(id))
                    .execute();
        }
    }

    @Override
    public Map<String, Set<Role>> getAllLoginRoles() {
        return dsl.selectFrom(LoginsRoles.TABLE)
                .fetch()
                .stream()
                .filter(JooqRoleService::roleExists)
                .collect(Collectors.groupingBy(
                        record -> record.get(LoginsRoles.LOGIN),
                        Collectors.mapping(JooqRoleService::readRole, Collectors.toSet())
                ));
    }

    @Override
    public List<Role> getRoles(String login) {
        return Caching.fetch(rolesCache, login, () -> dsl
                .selectFrom(LoginsRoles.TABLE)
                .where(LoginsRoles.LOGIN.equal(login))
                .fetch()
                .stream()
                .filter(JooqRoleService::roleExists)
                .map(JooqRoleService::readRole)
                .collect(Collectors.toList())
        );
    }

    private static Role readRole(Record record) {
        return Roles.INSTANCE.fromId(record.get(LoginsRoles.ROLE));
    }

    private static Boolean roleExists(Record record) {
        return Roles.INSTANCE.hasId(record.get(LoginsRoles.ROLE));
    }

    private final static class LoginsRoles {
        private static final String TABLE_NAME = "api_idm_login_roles";
        private static final Table<Record> TABLE = DSL.table(TABLE_NAME);
        private static final Field<String> LOGIN = DSL.field(DSL.name(TABLE_NAME, "login"), String.class);
        private static final Field<String> ROLE = DSL.field(DSL.name(TABLE_NAME, "role"), String.class);

        private LoginsRoles() {

        }
    }

    private final static class Logins {
        private static final String TABLE_NAME = "api_idm_logins";
        private static final Table<Record> TABLE = DSL.table(TABLE_NAME);
        private static final Field<String> LOGIN = DSL.field(DSL.name(TABLE_NAME, "login"), String.class);

        private Logins() {

        }
    }
}
