package ru.yandex.crypta.clients.reactor;

import java.net.URI;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import okhttp3.HttpUrl;
import okhttp3.Request;
import okhttp3.Response;

import ru.yandex.crypta.clients.reactor.artifact.ArtifactCreateRequest;
import ru.yandex.crypta.clients.reactor.artifact.ArtifactInstantiationResponse;
import ru.yandex.crypta.clients.reactor.artifact.KeyValueAttributes;
import ru.yandex.crypta.clients.reactor.artifact.LastArtifactInstanceResponse;
import ru.yandex.crypta.clients.reactor.artifact.NamespaceIdentifier;
import ru.yandex.crypta.clients.utils.RestClientWithOauth;
import ru.yandex.crypta.lib.proto.TReactorConfig;


public class RestReactorClient extends RestClientWithOauth implements ReactorClient {

    private final HttpUrl artifactInstantiateUrl;
    private final HttpUrl artifactInstanceCreateUrl;
    private final HttpUrl lastArtifactInstanceUrl;

    private static HttpUrl.Builder getApiUrl(String url) {
        return new HttpUrl.Builder().scheme("https").host(url).addPathSegments("api/v1");
    }

    private HttpUrl getArtifactInstantiateUrl(HttpUrl.Builder apiUrl) {
        return apiUrl.addPathSegments("a/i/instantiate").build();
    }

    private HttpUrl getArtifactInstanceCreateUrl(HttpUrl.Builder apiUrl) {
        return apiUrl.addPathSegments("a/i/create").build();
    }

    private HttpUrl getLastArtifactInstanceUrl(HttpUrl.Builder apiUrl) {
        return apiUrl.addPathSegments("a/i/get/last").build();
    }

    @Inject
    public RestReactorClient(TReactorConfig reactor)  {
        super(reactor.getToken(), true);
        String host = URI.create(reactor.getUrl()).getHost();
        this.artifactInstantiateUrl = getArtifactInstantiateUrl(getApiUrl(host));
        this.artifactInstanceCreateUrl = getArtifactInstanceCreateUrl(getApiUrl(host));
        this.lastArtifactInstanceUrl = getLastArtifactInstanceUrl(getApiUrl(host));
    }

    @Override
    public ArtifactInstantiationResponse ArtifactInstantiation(
        NamespaceIdentifier artifactIdentifier,
        String artifactType,
        String userTimestamp, // null or format like "2021-02-23T00:00:00.000"
        KeyValueAttributes attributes,
        boolean createIfNotExist
    ) {
        ObjectNode payload = getParameters(artifactIdentifier, artifactType, userTimestamp, attributes,
                createIfNotExist, null);

        Request request = request()
                .header("Accept", "application/json").header("Content-Type", "application/json")
                .url(this.artifactInstantiateUrl)
                .post(body(payload))
                .build();

        try (Response response = checkResponse(callWithRetry(request, 5))) {
            JsonNode json = readJson(response);
            return new ArtifactInstantiationResponse(json);
        }

    }

    @Override
    public ArtifactInstantiationResponse ArtifactInstanceCreate(
            NamespaceIdentifier artifactIdentifier,
            String artifactType,
            String userTimestamp, // null or format like "2021-02-23T00:00:00.000"
            KeyValueAttributes attributes,
            boolean createIfNotExist,
            ArtifactCreateRequest artifactCreateRequest
    ) {
        ObjectNode payload = getParameters(artifactIdentifier, artifactType, userTimestamp, attributes,
                false, artifactCreateRequest);

        Request request = request()
                .header("Accept", "application/json").header("Content-Type", "application/json")
                .url(this.artifactInstanceCreateUrl)
                .post(body(payload))
                .build();

        try (Response response = checkResponse(callWithRetry(request, 5))) {
            JsonNode json = readJson(response);
            return new ArtifactInstantiationResponse(json);
        }
    }

    @Override
    public LastArtifactInstanceResponse GetLastArtifactInstance(NamespaceIdentifier artifactIdentifier) {
        ObjectNode payload = getParameters(artifactIdentifier, null, null, null,
                false, null);

        Request request = request()
                .header("Accept", "application/json").header("Content-Type", "application/json")
                .url(this.lastArtifactInstanceUrl)
                .post(body(payload))
                .build();

        try (Response response = checkResponse(callWithRetry(request, 5))) {
            JsonNode json = readJson(response);
            return new LastArtifactInstanceResponse(json);
        }

    }

    private ObjectNode getParameters(
            NamespaceIdentifier artifactIdentifier,
            String artifactType,
            String userTimestamp,
            KeyValueAttributes attributes,
            boolean createIfNotExist,
            ArtifactCreateRequest artifactCreateRequest
    ) {
        ObjectMapper mapper = new ObjectMapper();
        ObjectNode payload = mapper.createObjectNode();

        payload.set("artifactIdentifier", artifactIdentifier.getRootNone());

        if (attributes != null) {
            payload.set("attributes", attributes.getRootNone());
        }

        if (artifactType != null) {
            ObjectNode metadata = payload.putObject("metadata");
            metadata.put("@type", artifactType);
        }

        if (userTimestamp != null) {
            payload.put("userTimestamp", userTimestamp);
        }

        if (createIfNotExist) {
            payload.put("createIfNotExist", true);
        }

        if (artifactCreateRequest != null) {
            payload.set("artifactCreateRequest", artifactCreateRequest.getRootNone());
        }

        return payload;
    }
}
