package ru.yandex.crypta.clients.sandbox;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.inject.Inject;

import okhttp3.HttpUrl;
import okhttp3.Request;

import ru.yandex.crypta.clients.sandbox.entity.Context;
import ru.yandex.crypta.clients.sandbox.entity.Id;
import ru.yandex.crypta.clients.sandbox.entity.Response;
import ru.yandex.crypta.clients.sandbox.entity.Task;
import ru.yandex.crypta.clients.sandbox.entity.TaskDraft;
import ru.yandex.crypta.clients.utils.RestClientWithOauth;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.proto.TSandboxConfig;

public class RestSandboxClient extends RestClientWithOauth implements SandboxClient {

    private final String url;

    private final String proxyUrl;

    @Inject
    public RestSandboxClient(TSandboxConfig sandbox) {
        super(sandbox.getToken());

        this.url = sandbox.getUrl();
        this.proxyUrl = sandbox.getProxyUrl();
    }

    @Override
    public Task create(TaskDraft draft) {
        Request request = request()
                .url(taskUrl().build())
                .post(body(draft))
                .build();
        try (var response = call(request)) {
            return read(response, Task.class);
        }
    }

    @Override
    public Task get(Id id) {
        Request request = request()
                .url(taskUrl(id).build())
                .get()
                .build();
        try (var response = call(request)) {
            return read(response, Task.class);
        }
    }

    @Override
    public Context getContext(Id id) {
        Request request = request()
                .url(taskUrl(id).addPathSegment("context").build())
                .get()
                .build();
        try (var response = call(request)) {
            return read(response, Context.class);
        }
    }

    @Override
    public Id start(Id id) {
        Map<String, List<Id>> command =
                Collections.singletonMap("id", Collections.singletonList(id));
        Request startRequest = request()
                .url(operationUrl("start").build())
                .put(body(command))
                .build();
        Response[] responses;
        try (var httpResponse = call(startRequest)) {
            responses = read(httpResponse, Response[].class);
        }
        Response theResponse = responses[0];
        if (theResponse.getStatus().equals("ERROR")) {
            throw Exceptions.illegal(theResponse.getMessage());
        }
        return theResponse.getId();
    }

    @Override
    public byte[] getLatest(String resource, String file) {
        Request request = request()
                .url(proxyUrl().addPathSegment("last").addPathSegment(resource).addPathSegment(file).build())
                .get()
                .build();
        return tryCall(request);
    }

    @Override
    public byte[] getLatest(String resource) {
        Request request = request()
                .url(proxyUrl().addPathSegment("last").addPathSegment(resource).build())
                .get()
                .build();
        return tryCall(request);
    }

    private byte[] tryCall(Request request) {
        try (var response = call(request)){

            if (!response.isSuccessful()) {
                throw Exceptions.unavailable();
            }

            return Objects.requireNonNull(response.body()).bytes();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    private HttpUrl.Builder taskUrl() {
        return url().addPathSegment("task");
    }

    private HttpUrl.Builder taskUrl(Id id) {
        return taskUrl().addPathSegment(String.valueOf(id.getValue()));
    }

    private HttpUrl.Builder operationUrl(String operation) {
        return url().addPathSegment("batch").addPathSegment("tasks").addPathSegment(operation);
    }

    private HttpUrl.Builder url() {
        return new HttpUrl.Builder().scheme("https").host(url).addPathSegment("api").addPathSegment("v1.0");
    }

    private HttpUrl.Builder proxyUrl() {
        return new HttpUrl.Builder().scheme("https").host(proxyUrl);
    }
}
