package ru.yandex.crypta.clients.staff;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import okhttp3.HttpUrl;
import okhttp3.Request;

import ru.yandex.crypta.clients.utils.RestClientWithOauth;
import ru.yandex.crypta.lib.proto.TStaffConfig;


public class RestStaffClient extends RestClientWithOauth implements StaffClient {
    private final String url;

    @Inject
    public RestStaffClient(TStaffConfig staffConfig) {
        super(staffConfig.getOauthToken());
        this.url = staffConfig.getUrl();
    }

    @Override
    public List<JsonNode> getAllPersons() {
        Request request = compileGetRequest(personsUrl().build());
        try (var response = call(request)) {
            var json = read(response, JsonNode.class);
            return processPages(json);
        }
    }

    @Override
    public List<JsonNode> getAllPersons(Timestamp modifiedSince) {
        Request request = compileGetRequest(personsUrl().addQueryParameter("_query", String.format("_meta.modified_at>=%s", modifiedSince.getTime())).build());
        try (var response = call(request)) {
            var json = read(response, JsonNode.class);
            return processPages(json);
        }
    }

    private HttpUrl.Builder url() {
        return new HttpUrl.Builder().scheme("https").host(url).addPathSegment("v3");
    }

    private HttpUrl.Builder personsUrl() {
        return url()
                .addPathSegment("persons")
                .addQueryParameter("_limit", "10000")
                .addQueryParameter("official.is_dismissed", "false")
                .addQueryParameter("_fields", "yandex.login,accounts,login");
    }

    private Request compileGetRequest(HttpUrl url) {
        return request()
                .url(url)
                .get()
                .build();
    }

    private List<JsonNode> processPages(JsonNode json) {
        List<JsonNode> persons = new ArrayList<>(json.get("total").asInt());
        json.get("result").forEach(persons::add);

        while (json.get("links").has("next")) {
            Request request = compileGetRequest(HttpUrl.parse(json.get("links").get("next").asText()));
            try (var response = call(request)) {
                json = read(response, JsonNode.class);
                json.get("result").forEach(persons::add);
            }
        }
        return persons;
    }
}
