package ru.yandex.crypta.clients.step;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Lists;
import okhttp3.HttpUrl;
import okhttp3.Request;
import okhttp3.Response;

import ru.yandex.crypta.clients.step.entity.StepEvent;
import ru.yandex.crypta.clients.step.entity.StepEventDescription;
import ru.yandex.crypta.clients.utils.RestClientWithOauth;
import ru.yandex.crypta.lib.proto.TSandboxConfig;

import static java.util.stream.Collectors.toList;
import static ru.yandex.crypta.clients.utils.JsonUtils.jsonArrayToStream;

public class RestStepEventClient extends RestClientWithOauth implements StepEventClient {

    private static final HttpUrl STEP_URL = new HttpUrl.Builder().scheme("https").host("step.sandbox.yandex-team.ru").build();
    private static final HttpUrl STEP_EVENTS_API_URL = STEP_URL.newBuilder().addPathSegments("api/v1/events").build();
    private static final HttpUrl STEP_EVENTS_DESC_API_URL = STEP_URL.newBuilder().addPathSegments("api/v1/event_desc").build();

    @Inject
    public RestStepEventClient(TSandboxConfig sandbox) {
        super(sandbox.getToken(), true);
    }


    @Override
    public void registerEventDescription(StepEventDescription eventType) {
        Request request = request()
                .url(STEP_EVENTS_DESC_API_URL)
                .post(body(eventType))
                .build();
        callAndCheck(request);

    }

    @Override
    public void updateEventDescription(StepEventDescription eventType) {
        Request request = request()
                .url(STEP_EVENTS_DESC_API_URL.newBuilder().addPathSegment(eventType.getName()).build())
                .put(body(eventType))
                .build();
        callAndCheck(request);
    }

    @Override
    public Optional<JsonNode> getEventDescription(String eventType) {
        Request request = request()
                .url(STEP_EVENTS_DESC_API_URL.newBuilder().addPathSegment(eventType).build())
                .get()
                .build();

        try (Response response = call(request)) {
            if (response.code() == 404) {
                return Optional.empty();
            } else {
                return Optional.of(readJson(checkResponse(response)));
            }
        }
    }

    @Override
    public void deleteEventDescription(String eventType) {
        Request request = request()
                .url(STEP_EVENTS_DESC_API_URL.newBuilder().addPathSegment(eventType).build())
                .delete()
                .build();
        callAndCheck(request);
    }

    @Override
    public String createEvent(StepEvent event) {
        Map<String, List<StepEvent>> payload = Collections.singletonMap("events", Collections.singletonList(event));
        Request request = request()
                .url(STEP_EVENTS_API_URL)
                .post(body(payload))
                .build();

        try (Response response = checkResponse(call(request))) {
            JsonNode json = readJson(response);
            List<String> ids = Lists.newArrayList(json.get("ids"))
                    .stream()
                    .map(JsonNode::textValue)
                    .toList();
            return ids.get(0);
        }

    }

    @Override
    public Optional<JsonNode> getEventInfo(String eventId) {
        Request request = request()
                .url(STEP_EVENTS_API_URL.newBuilder().addPathSegment(eventId).build())
                .get()
                .build();

        try (Response response = call(request)) {
            if (response.code() == 404) {
                return Optional.empty();
            } else {
                return Optional.of(readJson(checkResponse(response)));
            }
        }
    }

    @Override
    public List<JsonNode> searchEventInfo(String eventType, Map<String, String> params, Integer limit) {
        var urlBuilder = STEP_EVENTS_API_URL.newBuilder().addQueryParameter("name", eventType);
        for (Entry<String, String> param : params.entrySet()) {
            urlBuilder.addQueryParameter("params__" + param.getKey(), param.getValue());
        }

        if (limit != null) {
            urlBuilder.addQueryParameter("limit", limit.toString());
        }

        Request request = request()
                .url(urlBuilder.build())
                .get()
                .build();

        try (Response response = checkResponse(call(request))) {
            JsonNode eventsList = readJson(response).get("result");
            return jsonArrayToStream(eventsList).collect(toList());
        }

    }
}


