package ru.yandex.crypta.clients.tvm;

import java.io.IOException;
import java.util.Optional;
import java.util.function.Function;

import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;

import ru.yandex.inside.passport.tvm2.TvmHeaders;

public class TvmOkHttpInterceptor implements Interceptor {

    private final TvmClient tvmClient;
    private final Function<Request, Optional<Integer>> dstTvmIdSupplier;

    /**
     * Requests are served with dst tvm client id, based on request
     */
    public TvmOkHttpInterceptor(TvmClient tvmClient,
                                Function<Request, Optional<Integer>> dstTvmIdSupplier) {
        this.tvmClient = tvmClient;
        this.dstTvmIdSupplier = dstTvmIdSupplier;
    }

    /**
     * All requests are served with single dst tvm client id
     */
    public TvmOkHttpInterceptor(TvmClient tvmClient, int dstTvmId) {
        this(tvmClient, (request) -> Optional.of(dstTvmId));
    }

    @Override
    public Response intercept(Chain chain) throws IOException {
        Request request = chain.request();

        if (request.headers(TvmHeaders.SERVICE_TICKET).size() == 0) {
            Optional<Integer> dstTvmIdO = dstTvmIdSupplier.apply(request);

            if (dstTvmIdO.isPresent()) {
                Integer dstTvmId = dstTvmIdO.get();
                String serviceTicket = tvmClient.getServiceTicket(dstTvmId).orElseThrow(() ->
                        new IllegalStateException("Can't obtain tvm service ticket for tvmId=" + dstTvmId)
                );
                request = request.newBuilder().header(TvmHeaders.SERVICE_TICKET, serviceTicket).build();
            }
        }
        return chain.proceed(request);
    }
}
