package ru.yandex.crypta.clients.utils;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.misc.thread.ThreadUtils;

public abstract class RestClientWithOauth {

    private static final Logger LOG = LoggerFactory.getLogger(RestClientWithOauth.class);

    private static final MediaType JSON_UTF_8 = MediaType.parse("application/json; charset=utf-8");
    private final String oauthToken;
    private final OkHttpClient httpClient;
    private final ObjectMapper objectMapper = new ObjectMapper();

    protected RestClientWithOauth(String oauthToken) {
        this(oauthToken, true);
    }

    protected RestClientWithOauth(String oauthToken, boolean unsafeHttpClient) {
        this.oauthToken = oauthToken;
        if (unsafeHttpClient) {
            httpClient = OkHttpUtils.getUnsafeOkHttpClient();
        } else {
            httpClient = new OkHttpClient.Builder()
                    .readTimeout(5, TimeUnit.SECONDS)
                    .build();
        }
    }


    protected <T> RequestBody body(T entity) {
        return RequestBody.create(JSON_UTF_8, write(entity));
    }

    protected Request.Builder request() {
        return new Request.Builder().header("Authorization", "OAuth " + oauthToken);
    }

    private <T> String write(T task) {
        try {
            return objectMapper.writeValueAsString(task);
        } catch (JsonProcessingException e) {
            throw Exceptions.unchecked(e);
        }
    }

    protected okhttp3.Response call(Request request) {
        try {
            return httpClient.newCall(request).execute();
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    protected okhttp3.Response callAndCheck(Request request) {
        try {
            return checkResponse(httpClient.newCall(request).execute());
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    protected <T> T read(okhttp3.Response response, Class<T> theClass) {
        try {
            String body = Objects.requireNonNull(response.body()).string();
            if (!response.isSuccessful()) {
                throw new RuntimeException(body);
            }
            return objectMapper.readValue(body, theClass);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    protected JsonNode readJson(Response response) {
        try {
            String jsonData = Objects.requireNonNull(response.body()).string();
            return objectMapper.readTree(jsonData);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }

    }

    protected Response checkResponse(Response response) {
        if (response.isSuccessful()) {
            return response;
        } else {
            try {
                String responseBody = Objects.requireNonNull(response.body()).string();
                throw Exceptions.illegal(response.toString() + ":\n" + responseBody);
            } catch (IOException e) {
                throw new IllegalStateException(response.toString(), e);
            }

        }
    }

    protected Response callWithRetry(Request request, int retries) {
        Response response = null;

        while (retries > 0) {
            retries -= 1;
            try {
                response = httpClient.newCall(request).execute();
                break;
            }  catch (Exception e) {
                LOG.error("Last retry to call {} failed, fall asleep for 1 second", request.url());
                ThreadUtils.sleep(Duration.standardSeconds(1));
            }
        }
        if (response == null) {
            response = call(request);
        }

        return response;
    }
}
