package ru.yandex.crypta.clients.yabs;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.script.ScriptException;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.utils.OkHttpUtils;
import ru.yandex.crypta.clients.utils.OnlyInet6AddressDns;
import ru.yandex.crypta.common.exception.Exceptions;

import static ru.yandex.crypta.clients.utils.HttpExceptions.checkResponse;


public class HttpYabsClient implements YabsClient {


    private static final String AN_YANDEX_URL = "an.yandex.ru";
    private static final String DEBUG_COOKIE_SERVER = "bsinfo.yandex.ru";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private static final Logger LOG = LoggerFactory.getLogger(HttpYabsClient.class);

    private final OkHttpClient adsClient = new OkHttpClient.Builder()
            .readTimeout(10, TimeUnit.SECONDS)
            .build();

    private final OkHttpClient debugCookieClient = new OkHttpClient.Builder()
            .dns(new OnlyInet6AddressDns())
            .readTimeout(10, TimeUnit.SECONDS)
            .build();

    private Request.Builder buildBasicAdRequest(String pageId, Map<String, String> queryParams, String userId,
                                                String userAgent) {
        HttpUrl.Builder url = new HttpUrl.Builder()
                .scheme("https")
                .host(AN_YANDEX_URL)
                .addPathSegment("meta")
                .addPathSegment(pageId);
        queryParams.forEach(url::addQueryParameter);

        return new Request.Builder()
                .url(url.build())
                .header(Headers.USER_AGENT, userAgent)
                .header(Headers.COOKIE, userId);
    }

    private Request getAdRequest(String pageId, Map<String, String> queryParams, String userId, String userAgent,
                                 String yabsExpSid, DebugMode debugMode, String ip) {
        Request.Builder rb = buildBasicAdRequest(pageId, queryParams, userId, userAgent);

        if (!(debugMode == null)) {
            switch (debugMode) {
                case SIMPLE:
                    rb.header(Headers.COOKIE, userId + "; yabs-exp-sid=" + yabsExpSid);
                    break;
                case ADVANCED:
                case MX:
                    if (!Objects.equals(yabsExpSid, "")) {
                        rb.header(Headers.X_YABS_DEBUG_TOKEN, yabsExpSid);
                    }
                    rb.header(Headers.X_YABS_DEBUG_OUTPUT, YabsOutputFormat.JSON.value);
                    break;
            }
            if (debugMode.equals(DebugMode.MX)) {
                var mxDebugOptions = "{\"mx\": true, \"logs\": true, \"match_log\": true, \"parse_constants\": true}";
                rb.header(Headers.X_YABS_DEBUG_OPTIONS_JSON, mxDebugOptions);
            } else {
                var usualDebugOptions = "{\"logs\": true, \"match_log\": true, \"parse_constants\": true}";
                rb.header(Headers.X_YABS_DEBUG_OPTIONS_JSON, usualDebugOptions);
            }
        }

        // TODO: support setting the seed
        // rb.header(Headers.X_YABS_TEST_RANDOM, "1000");

        rb.header(Headers.X_FORWARDED_FOR, ip);

        return rb.get().build();
    }

    private Object decodeEntity(JsonNode entity) {
        if (entity.has("$data")) {
            String bodyData = entity.get("$data").toString();
            byte[] bodyBytes = Base64.getDecoder().decode(bodyData.substring(1, bodyData.length() - 1));

            return new String(bodyBytes, StandardCharsets.UTF_8);
        } else {
            return entity.asText();
        }
    }

    @Override
    public JsonNode getAdDebug(
            String pageId, Map<String, String> queryString, String yandexuid, String userAgent, String yabsExpSid,
            String ip
    ) {
        Request adRequest = getAdRequest(
                pageId, queryString, yandexuid, userAgent, yabsExpSid, DebugMode.MX, ip
        );
        try (Response response = adsClient.newCall(adRequest).execute()) {
            if (response.code() == OkHttpUtils.NOT_FOUND) {
                throw Exceptions.notFound();
            }

            JsonNode objectNode = OBJECT_MAPPER.readTree(Objects.requireNonNull(response.body()).string());

            Map<String, Object> responseMap = new HashMap<>();
            responseMap.put("body", decodeEntity(objectNode.get("http").get("entity")));
            responseMap.put("status", response.code());
            responseMap.put("url", adRequest.url().toString());
            responseMap.put("headers", adRequest.headers().toMultimap());
            responseMap.put(Fields.AD_DEBUG, objectNode);

            return OBJECT_MAPPER.convertValue(responseMap, JsonNode.class);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }


    /**
     * Extended banner log traversal. Finds banner fields in recursively kept
     * 'exts' fields.
     *
     * @param exts             Exts section of banner json
     * @param bannerDebugAccum Container to keep each banner info
     * @param wideBannerText   Banner texts to append to specific banner in bannerDebugAccum
     * @return fulfilled map of debug field per banner
     */
    private Map<String, Map<String, Object>> extractExtsLog(
            JsonNode exts,
            Map<String, Map<String, Object>> bannerDebugAccum,
            Map<String, String> wideBannerText) {
        if (exts.size() > 0) {
            exts.forEach(ext -> {
                if (ext.has(Fields.RESPONSE)) {
                    JsonNode response = ext.get(Fields.RESPONSE);
                    if (response.has(Fields.AD_LOGS)) {
                        if (response.get(Fields.AD_LOGS).has(Fields.AD_WIDE)) {
                            response.get(Fields.AD_LOGS).get(Fields.AD_WIDE).forEach(item -> {
                                Map<String, Object> bannerInfo = new HashMap<>();
                                bannerInfo
                                        .put(Fields.AD_BANNER_ID, item.get(Fields.AD_BANNER_ID).toString());
                                bannerInfo.put(Fields.AD_CTR, item.get(Fields.AD_CTR).toString());
                                bannerInfo.put(Fields.AD_COST, item.get(Fields.AD_COST).toString());
                                bannerInfo
                                        .put(Fields.AD_SHOW_TIME, item.get(Fields.AD_SHOW_TIME).toString());
                                bannerInfo
                                        .put(Fields.AD_PHRASE_ID, item.get(Fields.AD_PHRASE_ID).toString());
                                bannerInfo.put(Fields.AD_QUERY_ID, item.get(Fields.AD_QUERY_ID).toString());
                                bannerInfo.put("QueryIDHex",
                                        Long.toHexString(item.get(Fields.AD_QUERY_ID).asLong()));
                                bannerInfo.put(Fields.AD_SELECT_TYPE,
                                        item.get(Fields.AD_SELECT_TYPE).toString());
                                bannerInfo.put(Fields.AD_CATEGORY_ID,
                                        item.get(Fields.AD_CATEGORY_ID).toString());
                                bannerInfo.put(Fields.AD_BID_CORRECTION,
                                        item.get(Fields.AD_BID_CORRECTION).toString());
                                bannerInfo.put(Fields.AD_PHRASE_PLUS_WORDS,
                                        wideBannerText.get(item.get(Fields.AD_BANNER_ID).toString()));

                                bannerDebugAccum.put(item.get(Fields.AD_BANNER_ID).toString(), bannerInfo);
                            });
                        }
                    }

                    if (response.has(Fields.EXTS)) {
                        bannerDebugAccum
                                .putAll(extractExtsLog(response.get(Fields.EXTS), bannerDebugAccum,
                                        wideBannerText));
                    }
                }
            });
        }

        return bannerDebugAccum;
    }


    private JsonNode extractMx(JsonNode debug, Set<String> bannerIds) {
        Map<String, JsonNode> mx = new HashMap<>();
        if (debug.has(Fields.ML_MODELS)) {
            mx.put("common", debug.get(Fields.ML_MODELS).get(Fields.MX));
        }

        if (debug.has(Fields.EXTS)) {
            Map<String, List<JsonNode>> forBanners = new HashMap<>();
            debug.get(Fields.EXTS).forEach(ext -> {
                if (ext.has(Fields.RESPONSE)) {
                    JsonNode response = ext.get(Fields.RESPONSE);
                    if (response.has(Fields.ML_MODELS)) {
                        JsonNode mlModels = response.get(Fields.ML_MODELS);
                        if (mlModels.has(Fields.MX)) {
                            mlModels.get(Fields.MX).forEach(model -> {
                                String bannerId = model.get("banner_id").asText();

                                if (bannerIds.contains(bannerId)) {
                                    if (!forBanners.containsKey(bannerId)) {
                                        forBanners.put(bannerId, new ArrayList<>());
                                    }
                                    forBanners.get(bannerId).add(model);
                                }
                            });

                        }
                    }
                }
            });
            if (!forBanners.isEmpty()) {
                mx.put("for_banners", OBJECT_MAPPER.convertValue(forBanners, JsonNode.class));
            }
        }

        return OBJECT_MAPPER.convertValue(mx, JsonNode.class);
    }

    @Override
    public JsonNode getAdJson(String pageId, Map<String, String> queryString, String yandexuid, String userAgent,
                              String yabsExpSid, DebugMode debugMode, String ip) {
        JsonNode ad = getAd(pageId, queryString, yandexuid, userAgent, yabsExpSid, debugMode, ip);

        String rawAd;
        ObjectNode adObject = (ObjectNode) ad;

        if (ad.has("body")) {
            rawAd = ad.get("body").asText();
            try {
                JsonNode jsonDirectAd = AdParser.parse(rawAd);

                if (jsonDirectAd.has("direct")) {
                    JsonNode jsonAd = jsonDirectAd.get("direct");
                    adObject.set(Fields.JSON_BODY,
                            OBJECT_MAPPER.convertValue(new UserAd(jsonAd, jsonAd.has("ads")), JsonNode.class));
                } else {
                    throw new IllegalArgumentException();
                }
            } catch (ScriptException | IllegalArgumentException e) {
                LOG.debug("Error parsing ad body for user {}:\n{}\n", yandexuid, rawAd);
                LOG.debug("Request: {}", ad.get("url"));
                adObject.set(Fields.JSON_BODY,
                        OBJECT_MAPPER.convertValue(new UserAd(null, false), JsonNode.class));
            } catch (IOException e) {
                LOG.debug("Error getting ad for user {}", yandexuid);
            }
        }

        OBJECT_MAPPER.configure(JsonParser.Feature.ALLOW_NON_NUMERIC_NUMBERS, true);
        return OBJECT_MAPPER.convertValue(adObject, JsonNode.class);
    }

    @Override
    public JsonNode getAd(String pageId, Map<String, String> queryString, String yandexuid, String userAgent,
                          String yabsExpSid, DebugMode debugMode, String ip) {
        Request adRequest = getAdRequest(
                pageId, queryString, yandexuid, userAgent, yabsExpSid, debugMode, ip
        );
        try (Response response = adsClient.newCall(adRequest).execute()) {
            if (response.code() == OkHttpUtils.NOT_FOUND) {
                throw Exceptions.notFound();
            }

            Object body;
            Map<String, Map<String, Object>> bannerDebug = new HashMap<>();
            JsonNode mx = OBJECT_MAPPER.convertValue(new HashMap<String, Object>(), JsonNode.class);
            String bodyStr = Objects.requireNonNull(response.body()).string();

            if (debugMode == null || response.header(Headers.X_YABS_DEBUG_OUTPUT) == null || debugMode
                    .equals(DebugMode.SIMPLE)) {
                body = bodyStr;
            } else if (debugMode.equals(DebugMode.ADVANCED) || debugMode.equals(DebugMode.MX)) {
                JsonNode objectNode = OBJECT_MAPPER.readTree(bodyStr);

                body = decodeEntity(objectNode.get("http").get("entity"));

                Map<String, String> wideBannerText = new HashMap<>();
                objectNode.get(Fields.AD_LOGS).get(Fields.AD_WIDE_BANNER_TEXT).forEach(item -> {
                    wideBannerText.put(item.get(Fields.AD_BANNER_ID).toString(),
                            item.get(Fields.AD_PHRASE_PLUS_WORDS).toString());
                });

                bannerDebug = extractExtsLog(objectNode.get(Fields.EXTS), bannerDebug, wideBannerText);

                for (JsonNode item : objectNode.get(Fields.AD_LOGS).get(Fields.AD_WIDE_BANNER_TEXT)) {
                    JsonNode plusWords;
                    if (item.has(Fields.AD_PHRASE_PLUS_WORDS)) {
                        plusWords = item.get(Fields.AD_PHRASE_PLUS_WORDS);
                    } else {
                        plusWords = OBJECT_MAPPER.convertValue("", JsonNode.class);
                    }

                    // Debug info is not always consistent; get the whole banner only
                    if (bannerDebug.containsKey(item.get(Fields.AD_BANNER_ID).toString())) {
                        bannerDebug.get(item.get(Fields.AD_BANNER_ID).toString())
                                .put(Fields.AD_PHRASE_PLUS_WORDS, plusWords);
                    }
                }

                if (debugMode.equals(DebugMode.MX)) {
                    // Use MX factors only for winner-banners,
                    // otherwise the response would be too large to display
                    mx = extractMx(objectNode, bannerDebug.keySet());
                }

            } else {
                body = OBJECT_MAPPER.readTree(bodyStr);
            }

            Map<String, Object> responseMap = new HashMap<>();
            responseMap.put("body", body);
            responseMap.put("status", response.code());
            responseMap.put("url", adRequest.url().toString());
            responseMap.put("headers", adRequest.headers().toMultimap());
            responseMap.put(Fields.AD_DEBUG, bannerDebug);
            responseMap.put("mx", mx);

            return OBJECT_MAPPER.convertValue(responseMap, JsonNode.class);
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public JsonNode updateDebugCookie() {
        Request aShotRequest = new Request.Builder()
                .url(new HttpUrl.Builder()
                        .scheme("http")
                        .host(DEBUG_COOKIE_SERVER)
                        .addPathSegment("debug_cookie.txt")
                        .build())
                .get()
                .build();
        try (Response response = debugCookieClient.newCall(aShotRequest).execute()) {
            checkResponse(response);
            return OBJECT_MAPPER.convertValue(Objects.requireNonNull(response.body()).string(), JsonNode.class);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
