package ru.yandex.crypta.lib.experiments;

import java.io.Closeable;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import NBSYeti.NExperimentParameters.Experiment.TExperimentParameters;
import NExperiments.Experiment.TExperiment;
import NExperiments.Experiment.TExperimentStats;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;

public class Experiments implements Closeable {

    private static final String NOT_AVAILABLE = "N/A";

    static {
        System.loadLibrary("experiments-java");
    }

    private final Map<Long, String> descriptions = new HashMap<>();
    private final Map<Long, String> issues = new HashMap<>();

    private volatile long handle;

    public Experiments(byte[] jsonData, byte[] abJsonData) {
        this.handle = load(jsonData, abJsonData);

        ArrayNode experimentParameters = (ArrayNode) parseExperimentParameters(jsonData);

        for (JsonNode experimentParameter : experimentParameters) {
            long experimentId = experimentParameter.get("ExperimentID").asLong();
            descriptions.put(experimentId, experimentParameter.get("Description").asText());
            issues.put(experimentId, experimentParameter.get("JiraID").asText());
        }
    }

    private static JsonNode parseExperimentParameters(byte[] jsonData) {
        try {
            return new ObjectMapper().readTree(jsonData);
        } catch (IOException e) {
            throw new RuntimeException("Unable to parse experiments parameters");
        }
    }

    public TExperimentParameters getExperimentParameters(long uniqId, long timestamp) {
        assertInitialized();
        return getExperimentParameters(handle, uniqId, timestamp);
    }

    public List<TExperiment> getActiveExperiments(long uniqid, long timestamp) {
        assertInitialized();

        return getActiveExperiments(handle, uniqid, timestamp)
                .stream().map(experiment -> TExperiment.newBuilder().mergeFrom(experiment)
                        .setDescription(descriptions.getOrDefault(experiment.getExperimentID(), NOT_AVAILABLE))
                        .setIssue(issues.getOrDefault(experiment.getExperimentID(), NOT_AVAILABLE))
                        .build())
                .collect(Collectors.toList());
    }

    private void assertInitialized() {
        if (handle == 0) {
            throw new RuntimeException("Object is not initialized correctly");
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (handle != 0) {
            destroy(handle);
            handle = 0;
        }
    }

    public static TExperimentStats getExperimentStats(String command, String ytToken) {
        return getExperimentStatsNative(command, ytToken);
    }

    private static native long load(byte[] jsonData, byte[] abJsonData);

    private static native void destroy(long handle);

    private static native TExperimentParameters getExperimentParameters(long handle, long uniqId, long timestamp);

    private static native List<TExperiment> getActiveExperiments(long handle, long uniqId, long timestamp);

    private static native TExperimentStats getExperimentStatsNative(String command, String ytToken);

}
