package ru.yandex.crypta.lib.getoptpb;

import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Predicate;

import NGetoptPb.Confoption;
import com.google.common.base.CaseFormat;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import joptsimple.OptionParser;

import ru.yandex.crypta.lib.proto.Extensions;

public class GetoptPb {

    public static final String DEVELOP = "develop";
    public static final String TESTING = "testing";
    public static final String PRODUCTION = "production";
    public static final String STABLE = "stable";

    public static void handleArgs(String[] args, Message.Builder message) {
        var parser = new OptionParser();
        message.getDescriptorForType()
                .getFields()
                .forEach(addOption("", parser));
        var help = parser.accepts("help").forHelp();
        var optionSet = parser.parse(args);
        if (optionSet.has(help)) {
            try {
                parser.printHelpOn(System.out);
            } catch (IOException e) {
                e.printStackTrace(System.err);
            }
            System.exit(1);
        }
        processExtensions(message);
    }

    private static Consumer<Descriptors.FieldDescriptor> addOption(String prefix, OptionParser optionParser) {
        return (each) -> {
            var isMessage = each.getType().equals(Descriptors.FieldDescriptor.Type.MESSAGE);
            var eachName = CaseFormat.UPPER_CAMEL.converterTo(CaseFormat.LOWER_HYPHEN).convert(each.getName());
            var realPrefix = prefix.isEmpty() ? "" : (prefix + "-");
            if (isMessage) {
                var nextPrefix = realPrefix + eachName;
                for (var subField : each.getMessageType().getFields()) {
                    addOption(nextPrefix, optionParser).accept(subField);
                }
            } else {
                var name = realPrefix + eachName;
                optionParser.accepts(name);
            }
        };
    }

    public static Optional<Descriptors.EnumValueDescriptor> findEnumValue(Descriptors.EnumDescriptor enumDescriptor,
            String value)
    {
        return enumDescriptor
                .getValues()
                .stream()
                .filter(matchesValue(value))
                .findFirst();
    }

    private static Predicate<Descriptors.EnumValueDescriptor> matchesValue(String value) {
        return each -> {
            if (value.equalsIgnoreCase(each.getName())) {
                return true;
            }
            if (each.getOptions().hasExtension(Confoption.val) &&
                    value.equalsIgnoreCase(each.getOptions().getExtension(Confoption.val)))
            {
                return true;
            }
            return false;
        };
    }

    private static String selectFromEnv(String key, Descriptors.FieldDescriptor field) {
        if (key.equalsIgnoreCase(DEVELOP)) {
            return field.getOptions().getExtension(Extensions.develop);
        } else if (key.equalsIgnoreCase(TESTING)) {
            return field.getOptions().getExtension(Extensions.testing);
        } else if (key.equalsIgnoreCase(PRODUCTION) || key.equalsIgnoreCase(STABLE)) {
            return field.getOptions().getExtension(Extensions.production);
        } else {
            return null;
        }
    }

    private static Optional<Integer> tryParseInteger(String value) {
        try {
            return Optional.of(Integer.parseInt(value));
        } catch (NumberFormatException ex) {
            return Optional.empty();
        }
    }

    private static String getSelectFromEnv(Descriptors.FieldDescriptor each) {
        return each.getOptions().getExtension(Extensions.selectFromEnv);
    }

    private static String getFromEnv(Descriptors.FieldDescriptor each) {
        return each.getOptions().getExtension(Extensions.fromEnv);
    }

    private static String getOrEmpty(Map<String, String> environment, String extensionValue) {
        return environment.getOrDefault(extensionValue, "");
    }

    public static void processExtensions(Message.Builder message) {
        for (var each : message.getDescriptorForType().getFields()) {
            var isMessage = each.getType().equals(Descriptors.FieldDescriptor.Type.MESSAGE);
            var isEnum = each.getType().equals(Descriptors.FieldDescriptor.Type.ENUM);
            var isString = each.getType().equals(Descriptors.FieldDescriptor.Type.STRING);
            var isBoolean = each.getType().equals(Descriptors.FieldDescriptor.Type.BOOL);
            // TODO support all numeric types types
            var isNumber = each.getType().equals(Descriptors.FieldDescriptor.Type.INT32);
            var isNotRepeated = !each.isRepeated();
            var hasFromEnv = each.getOptions().hasExtension(Extensions.fromEnv);
            var hasSelectFromEnv = each.getOptions().hasExtension(Extensions.selectFromEnv);
            var hasFlagFromEnv = each.getOptions().hasExtension(Extensions.flagFromEnv);
            var environment = System.getenv();

            if (isMessage) {
                Message.Builder subMessage = message.getFieldBuilder(each);
                processExtensions(subMessage);
                continue;
            }
            if (isBoolean && isNotRepeated && hasFlagFromEnv) {
                String extensionValue = each.getOptions().getExtension(Extensions.flagFromEnv);
                if (environment.containsKey(extensionValue)) {
                    message.setField(each, true);
                    continue;
                }
            }
            if (isEnum && isNotRepeated) {
                if (hasFromEnv) {
                    String extensionValue = getFromEnv(each);
                    String stringValue = getOrEmpty(environment, extensionValue);
                    var value = findEnumValue(each.getEnumType(), stringValue);
                    if (value.isPresent()) {
                        message.setField(each, value.get());
                        continue;
                    }
                }
            }
            if (isString && isNotRepeated) {
                if (hasFromEnv) {
                    String extensionValue = getFromEnv(each);
                    String value = getOrEmpty(environment, extensionValue);
                    if (!value.isEmpty()) {
                        message.setField(each, value);
                        continue;
                    }
                }
                if (hasSelectFromEnv) {
                    String extensionValue = getSelectFromEnv(each);
                    String key = getOrEmpty(environment, extensionValue);
                    String value = selectFromEnv(key, each);

                    if (Objects.nonNull(value)) {
                        message.setField(each, value);
                        continue;
                    }
                }
            }
            if (isNumber && isNotRepeated) {
                if (hasFromEnv) {
                    String extensionValue = getFromEnv(each);
                    if (environment.containsKey(extensionValue)) {
                        var intValue = tryParseInteger(environment.get(extensionValue));
                        if (intValue.isPresent()) {
                            message.setField(each, intValue.get());
                            continue;
                        }
                    }
                }
                if (hasSelectFromEnv) {
                    String extensionValue = getSelectFromEnv(each);
                    String key = getOrEmpty(environment, extensionValue);
                    String value = selectFromEnv(key, each);

                    if (Objects.nonNull(value)) {
                        var intValue = tryParseInteger(value);
                        if (intValue.isPresent()) {
                            message.setField(each, intValue.get());
                            continue;
                        }
                    }
                }
            }
        }
    }

}
