package ru.yandex.crypta.graph;

import java.util.Objects;

import com.google.protobuf.InvalidProtocolBufferException;

import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.TGenericID;

public class Identifier {
    static {
        System.loadLibrary("identifiers-java");
    }

    protected String value;
    protected int typeAsProtoEnumNumber;

    public Identifier(EIdType type, String value) {
        this.typeAsProtoEnumNumber = type.getNumber();
        this.value = value;
    }

    protected Identifier(int typeAsProtoEnumNumber, String value) {
        this.typeAsProtoEnumNumber = typeAsProtoEnumNumber;
        this.value = value;
    }

    public Identifier(String stringType, String value) {
        this.typeAsProtoEnumNumber = ConvertTypeToEnumCodeNative(stringType);
        this.value = value;
    }

    public TGenericID toTGenericID() throws InvalidProtocolBufferException {
        return TGenericID.parseFrom(toProto());
    }

    public String getValue() {
        return value;
    }

    public EIdType getType() {
        return EIdType.forNumber(typeAsProtoEnumNumber);
    }

    public boolean isValid() {
        return IsValidNative();
    }

    public String getNormalizedValue() {
        return NormalizeNative();
    }

    public String getMd5() {
        return GetMd5Native();
    }

    public String getSha256() {
        return GetSha256Native();
    }

    public String getNext() {
        return GetNextNative();
    }

    public byte[] toProto() {
        return ToProtoNative();
    }

    public static Identifier fromProto(byte[] protoBytes) {
        return FromProtoNative(protoBytes);
    }

    public static Identifier random(String stringType) {
        return new Identifier(stringType, new Identifier(stringType, "").getNext());
    }

    public static Identifier random(EIdType idType) {
        return new Identifier(idType, new Identifier(idType, "").getNext());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Identifier)) {
            return false;
        }
        Identifier that = (Identifier) o;
        return Objects.equals(value, that.value) && typeAsProtoEnumNumber == that.typeAsProtoEnumNumber;
    }

    @Override
    public int hashCode() {
        return Objects.hash(value, typeAsProtoEnumNumber);
    }

    protected native boolean IsValidNative();
    protected native int ConvertTypeToEnumCodeNative(String typeString);
    protected native String NormalizeNative();
    protected native String GetMd5Native();
    protected native String GetSha256Native();
    protected native String GetNextNative();
    protected native byte[] ToProtoNative();
    protected static native Identifier FromProtoNative(byte[] protoBytes);

    public static native void SetRandomSeed(int seed);
}
