#include "ru_yandex_crypta_graph_Identifier.h"
#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/proto/identifiers/identifiers.pb.h>
#include <util/generic/string.h>
#include <util/random/random.h>

void rethrow_as_java(JNIEnv* env) {
    try {
        throw;
    } catch (const std::exception& e) {
        jclass jc = env->FindClass("java/lang/RuntimeException");
        if (jc) {
            env->ThrowNew(jc, e.what());
        }
    } catch (...) {
        jclass jc = env->FindClass("java/lang/RuntimeException");
        if (jc) {
            env->ThrowNew(jc, "Unhandled exception");
        }
    }
}

NIdentifiers::TGenericID MakeGenericId(JNIEnv* env, jobject jo) {
    // locate java object instance
    const jclass javaClass = env->GetObjectClass(jo);
    if (javaClass == NULL)
        throw std::invalid_argument("bad javaClass");

    // get value
    const jfieldID jfValue = env->GetFieldID(javaClass, "value", "Ljava/lang/String;");
    if (jfValue == NULL)
        throw std::invalid_argument("bad method");

    jstring jsValue = (jstring)env->GetObjectField(jo, jfValue);
    const char* cValue = env->GetStringUTFChars(jsValue, NULL);
    TString sValue = cValue;
    env->ReleaseStringUTFChars(jsValue, cValue);

    // get type
    const jfieldID jfIdentifierType = env->GetFieldID(javaClass, "typeAsProtoEnumNumber", "I");
    if (jfIdentifierType == NULL)
        throw std::invalid_argument("bad field");
    const NCrypta::NIdentifiersProto::NIdType::EIdType identifierType{
        static_cast<NCrypta::NIdentifiersProto::NIdType::EIdType>(env->GetIntField(jo, jfIdentifierType))};

    // create native TGenericID
    return NIdentifiers::TGenericID{identifierType, sValue};
}

JNIEXPORT jboolean JNICALL Java_ru_yandex_crypta_graph_Identifier_IsValidNative(JNIEnv* env, jobject jo) {
    try {
        return MakeGenericId(env, jo).IsValid();
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }
}

JNIEXPORT jint JNICALL Java_ru_yandex_crypta_graph_Identifier_ConvertTypeToEnumCodeNative(JNIEnv* env, jobject jo, jstring jsIdentifierType) {
    try {
        Y_UNUSED(jo);
        const char* cIdentifierType = env->GetStringUTFChars(jsIdentifierType, NULL);
        TString sIdentifierType = cIdentifierType;
        env->ReleaseStringUTFChars(jsIdentifierType, cIdentifierType);

        return NIdentifiers::TGenericID(sIdentifierType).GetType();
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }
}

JNIEXPORT jstring JNICALL Java_ru_yandex_crypta_graph_Identifier_NormalizeNative(JNIEnv* env, jobject jo) {
    try {
        return env->NewStringUTF(MakeGenericId(env, jo).Normalize().c_str());
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }
}

JNIEXPORT jstring JNICALL Java_ru_yandex_crypta_graph_Identifier_GetMd5Native(JNIEnv* env, jobject jo) {
    try {
        NIdentifiers::TGenericID genericId = MakeGenericId(env, jo);
        return env->NewStringUTF(genericId.GetMd5().c_str());
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }
}

JNIEXPORT jstring JNICALL Java_ru_yandex_crypta_graph_Identifier_GetSha256Native(JNIEnv* env, jobject jo) {
    try {
        NIdentifiers::TGenericID genericId = MakeGenericId(env, jo);
        return env->NewStringUTF(genericId.GetSha256().c_str());
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }
}

JNIEXPORT jstring JNICALL Java_ru_yandex_crypta_graph_Identifier_GetNextNative(JNIEnv* env, jobject jo) {
    try {
        NIdentifiers::TGenericID genericId = MakeGenericId(env, jo);
        return env->NewStringUTF(genericId.GetNext().c_str());
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }
}

JNIEXPORT jbyteArray JNICALL Java_ru_yandex_crypta_graph_Identifier_ToProtoNative(JNIEnv* env, jobject jo) {
    try {
        NIdentifiers::TGenericID genericId = MakeGenericId(env, jo);

        auto serialized = genericId.ToProto().SerializeAsString();
        jbyteArray result = env->NewByteArray(serialized.size());
        env->SetByteArrayRegion(result, 0, serialized.size(), reinterpret_cast<const jbyte*>(serialized.c_str()));

        return result;
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }
}

JNIEXPORT jobject JNICALL Java_ru_yandex_crypta_graph_Identifier_FromProtoNative(JNIEnv* env, jclass jcls, jbyteArray data) {
    try {
        NCrypta::NIdentifiersProto::TGenericID proto;

        jboolean isCopy;
        jint length = env->GetArrayLength(data);
        jbyte* buf = env->GetByteArrayElements(data, &isCopy);
        TString dataString(reinterpret_cast<char const*>(buf), length);

        Y_PROTOBUF_SUPPRESS_NODISCARD proto.ParseFromString(dataString);
        env->ReleaseByteArrayElements(data, buf, JNI_ABORT);

        NIdentifiers::TGenericID genericId(proto);

        jmethodID constructor = env->GetMethodID(jcls, "<init>", "(ILjava/lang/String;)V");
        return env->NewObject(
            jcls, constructor,
            genericId.GetType(),
            env->NewStringUTF(genericId.GetValue().c_str())
        );
    } catch (...) {
        rethrow_as_java(env);
        return 0;
    }

}


JNIEXPORT void JNICALL Java_ru_yandex_crypta_graph_Identifier_SetRandomSeed(JNIEnv *, jclass, jint seed) {
    SetRandomSeed(seed);
}
