#include "ru_yandex_crypta_lib_lemmer_Lemmer.h"

#include <crypta/lib/native/lemmer/get_lemmas.h>

#include <util/generic/strbuf.h>
#include <util/generic/string.h>

namespace {
    void rethrow_as_java(JNIEnv* env) {
        try {
            throw;
        } catch (const std::exception& e) {
            jclass jc = env->FindClass("java/lang/RuntimeException");
            if (jc) {
                env->ThrowNew(jc, e.what());
            }
        } catch (...) {
            jclass jc = env->FindClass("java/lang/RuntimeException");
            if (jc) {
                env->ThrowNew(jc, "Unhandled exception");
            }
        }
    }

    void SetLang(TLangMask& langs, const char* language){
        auto lang = LanguageByName(language);
        if (lang != ELanguage::LANG_UNK && lang != ELanguage::LANG_EMPTY) {
            langs.Set(lang);
        }
    }
}

JNIEXPORT jobject JNICALL Java_ru_yandex_crypta_lib_lemmer_Lemmer_getLemmas(JNIEnv* env, jclass, jstring jInput, jstring jLang)
{
    try {
        if (!jInput || !jLang) {
            ythrow yexception() << "Missing arguments";
        }

        TLangMask langs = TLangMask(LANG_RUS, LANG_ENG);
        const char* charLang = env->GetStringUTFChars(jLang, NULL);
        SetLang(langs, charLang);
        env->ReleaseStringUTFChars(jLang, charLang);

        const char* charInput = env->GetStringUTFChars(jInput, NULL);
        const auto& lemmas = NCrypta::GetLemmas(TUtf16String::FromUtf8(charInput), langs);
        env->ReleaseStringUTFChars(jInput, charInput);

        auto hashSetClass = env->FindClass("java/util/HashSet");
        auto init = env->GetMethodID(hashSetClass, "<init>", "()V");
        auto hashSet = env->NewObject(hashSetClass, init);

        auto add = env->GetMethodID(hashSetClass, "add", "(Ljava/lang/Object;)Z");

        for (const auto& lemma: lemmas) {
            auto jLemma = env->NewStringUTF(WideToUTF8(lemma).c_str());
            env->CallObjectMethod(hashSet, add, jLemma);
            env->DeleteLocalRef(jLemma);
        }

        auto hashSetGobal = static_cast<jobject>(env->NewGlobalRef(hashSet));
        env->DeleteLocalRef(hashSet);
        env->DeleteLocalRef(hashSetClass);

        return hashSetGobal;
    } catch(...) {
        rethrow_as_java(env);
        return nullptr;
    }
}
