package ru.yandex.crypta.lib.mds;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import javax.inject.Inject;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;

import ru.yandex.crypta.lib.proto.TS3Config;

public class DefaultMdsService implements MdsService {
    private static final String BUCKET = "crypta";
    private static final String REGION = "yndx";
    private static final String SUFFIX = "/";
    private final AmazonS3 client;

    @Inject
    public DefaultMdsService(TS3Config s3) {
        BasicAWSCredentials credentials = new BasicAWSCredentials(s3.getAccessKeyId(), s3.getSecretKey());
        this.client = AmazonS3ClientBuilder.standard()
                .disableChunkedEncoding() // https://azalio.at.yandex-team.ru/3650
                .withCredentials(new AWSStaticCredentialsProvider(credentials))
                .withEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(s3.getEndpoint(), REGION))
                .withClientConfiguration(new ClientConfiguration()
                        .withConnectionTimeout(100)
                        .withRequestTimeout(10_000)
                        .withMaxErrorRetry(3)
                        .withMaxConnections(5)
                )
                .build();
    }

    @Override
    public String putFile(String folder, String key, byte[] data) {
        String fullKey = folder + SUFFIX + key;

        ObjectMetadata objectMetadata = new ObjectMetadata();
        objectMetadata.setContentLength((long) data.length);

        client.putObject(BUCKET,fullKey, new ByteArrayInputStream(data), objectMetadata);

        return client.getUrl(BUCKET, fullKey).getPath();
    }

    @Override
    public S3Object getFile(String fullKey) {
        return client.getObject(BUCKET, fullKey);
    }

    @Override
    public void deleteFile(String folder, String key) {
        client.deleteObject(BUCKET, folder + SUFFIX + key);
    }

    @Override
    public void createFolder(String folderName) {
        // Service method; remove after use
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentLength(0);
        InputStream emptyContent = new ByteArrayInputStream(new byte[0]);
        PutObjectRequest putObjectRequest = new PutObjectRequest(BUCKET,
                folderName + SUFFIX, emptyContent, metadata);

        client.putObject(putObjectRequest);
    }
}
