package ru.yandex.crypta.lib.schedulers;

import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import org.quartz.Job;
import org.quartz.JobBuilder;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.TriggerBuilder;
import org.quartz.impl.matchers.GroupMatcher;

import ru.yandex.crypta.common.exception.Exceptions;

public class QuartzScheduler {

    private final Scheduler scheduler;

    public QuartzScheduler(Scheduler scheduler) {
        this.scheduler = scheduler;
    }

    public Scheduler get() {
        return scheduler;
    }

    public TriggerBuilder startNow(String id) {
        return TriggerBuilder.newTrigger()
                .withIdentity(uuid(), id)
                .startNow();
    }

    public <T extends Job> JobBuilder job(String id, Class<T> clazz) {
        return JobBuilder.newJob()
                .withIdentity(uuid(), id)
                .ofType(clazz);
    }

    private String uuid() {
        return UUID.randomUUID().toString();
    }

    public List<JobDetail> getJobs(String group) {
        try {
            return scheduler.getJobKeys(GroupMatcher.groupEquals(group))
                     .stream()
                     .map(x -> {
                         try {
                             return scheduler.getJobDetail(x);
                         } catch (SchedulerException e) {
                             throw Exceptions.unchecked(e);
                         }
                     })
                     .collect(Collectors.toList());
        } catch (SchedulerException e) {
            throw Exceptions.unchecked(e);
        }
    }

    public void schedule(JobBuilder jobBuilder, TriggerBuilder triggerBuilder) {
        try {
            scheduler.scheduleJob(jobBuilder.build(), Collections.singleton(triggerBuilder.build()), true);
        } catch (SchedulerException e) {
            throw Exceptions.unchecked(e);
        }
    }

}
