package ru.yandex.crypta.lib.schedulers;

import java.util.Properties;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

import javax.inject.Inject;
import javax.sql.DataSource;

import org.glassfish.jersey.internal.inject.InjectionManager;
import org.quartz.Job;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.impl.StdSchedulerFactory;
import org.quartz.spi.JobFactory;

import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.lib.proto.TQuartzConfig;

public class Schedulers {

    private final Properties quartzProperties;
    private Scheduler quartz;
    private ScheduledExecutorService executor;
    private InjectionManager injectionManager;

    @Inject
    public Schedulers(
            PostgresClient postgresClient,
            InjectionManager injectionManager,
            TQuartzConfig quartzConfig
    )
    {
        this.quartzProperties = createQuartzProperties(quartzConfig, postgresClient.getDataSource());
        this.executor = Executors.newScheduledThreadPool(4);
        this.quartz = null;
        this.injectionManager = injectionManager;
    }

    public synchronized QuartzScheduler getQuartz() {
        if (quartz == null) {
            createQuartzScheduler();
        }
        return new QuartzScheduler(quartz);
    }

    public ScheduledExecutorService getExecutor() {
        return executor;
    }

    private void createQuartzScheduler() {
        try {
            StdSchedulerFactory schedulerFactory = new StdSchedulerFactory();
            schedulerFactory.initialize(quartzProperties);
            quartz = schedulerFactory.getScheduler();
            quartz.setJobFactory(injectingJobFactory());
            quartz.start();
            Runtime.getRuntime().addShutdownHook(shutdownHook());
        } catch (SchedulerException e) {
            throw new RuntimeException(e);
        }
    }

    private JobFactory injectingJobFactory() {
        return (bundle, scheduler) -> {
            JobDetail jobDetail = bundle.getJobDetail();
            Class<? extends Job> jobClass = jobDetail.getJobClass();
            return injectionManager.createAndInitialize(jobClass);
        };
    }

    private Thread shutdownHook() {
        return new Thread(() -> {
            try {
                executor.shutdown();
                quartz.shutdown();
            } catch (SchedulerException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private Properties createQuartzProperties(TQuartzConfig quartzConfig, DataSource dataSource)
    {
        String instanceName = "quartz";
        String suffix = quartzConfig.getSuffix();

        if (!(suffix == null || suffix.equals(""))) {
            instanceName = instanceName + "-" + suffix;
        }

        QuartzConnectionProviderSingleton.INSTANCE.put(dataSource);

        Properties properties = new Properties();
        properties.setProperty("org.quartz.scheduler.instanceName", instanceName);
        properties.setProperty("org.quartz.scheduler.instanceId", "AUTO");

        properties.setProperty("org.quartz.scheduler.rmi.export", "false");
        properties.setProperty("org.quartz.scheduler.rmi.proxy", "false");
        properties.setProperty("org.quartz.scheduler.wrapJobExecutionInUserTransaction", "false");

        properties.setProperty("org.quartz.threadPool.class", "org.quartz.simpl.SimpleThreadPool");
        properties.setProperty("org.quartz.threadPool.threadCount", "2");
        properties.setProperty("org.quartz.threadPool.threadPriority", "5");
        properties.setProperty("org.quartz.threadPool.threadsInheritContextClassLoaderOfInitializingThread", "true");

        properties.setProperty("org.quartz.jobStore.class", "org.quartz.impl.jdbcjobstore.JobStoreTX");
        properties.setProperty("org.quartz.jobStore.dataSource", "default");
        properties.setProperty("org.quartz.jobStore.driverDelegateClass",
                "org.quartz.impl.jdbcjobstore.PostgreSQLDelegate");
        properties.setProperty("org.quartz.jobStore.isClustered", "true");
        properties.setProperty("org.quartz.jobStore.clusterCheckinInterval", "20000");

        properties.setProperty("org.quartz.dataSource.default.connectionProvider.class",
                "ru.yandex.crypta.lib.schedulers.QuartzConnectionProvider");
        properties.setProperty("org.quartz.jobStore.misfireThreshold", "60000");
        properties.setProperty("org.quartz.plugin.triggerHistory.class",
                "org.quartz.plugins.history.LoggingTriggerHistoryPlugin");

        return properties;
    }
}
