package ru.yandex.crypta.service.dmp;

import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

public class YtDmpService implements DmpService {
    private final YtService yt;
    private final YPath pathDmp = YPath.simple("//home/crypta/production/dmp/out");
    private final YPath pathIndex = pathDmp.child("index");
    private final YPath pathMeta = pathDmp.child("meta");

    private final Cache<Integer, Map<String, DmpIndex>> indexCache = CacheBuilder.newBuilder()
            .expireAfterWrite(1, TimeUnit.HOURS)
            .build();

    private static final Logger LOG = LoggerFactory.getLogger(YtDmpService.class);

    private final ExecutorService parentExecutor = Executors.newSingleThreadExecutor();
    private final ListeningExecutorService executorService = MoreExecutors.listeningDecorator(parentExecutor);

    private final LoadingCache<String, Map<String, DmpMeta>> metaCache = CacheBuilder.newBuilder()
            .refreshAfterWrite(1, TimeUnit.HOURS)
            .build(
                    new CacheLoader<String, Map<String, DmpMeta>>() {
                        @Override
                        public Map<String, DmpMeta> load(String dmpLogin) {
                            return getMeta(dmpLogin);
                        }

                        @Override
                        public ListenableFuture<Map<String, DmpMeta>> reload(final String dmpLogin,
                                Map<String, DmpMeta> oldValue)
                        {
                            // Load new values asynchronously and non-blocking
                            return executorService.submit(() -> load(dmpLogin));
                        }
                    }
            );

    @Inject
    public YtDmpService(YtService yt) {
        this.yt = yt;
    }

    @Override
    public Map<String, DmpIndex> getIndex() {
        ListF<DmpIndex> result = Cf.arrayList();
        yt.getHahn().tables().read(pathIndex,
                YTableEntryTypes.yson(DmpIndex.class), (Consumer<DmpIndex>) result::add);
        return result.stream().collect(Collectors.toMap(
                item -> String.valueOf(item.getId()),
                item -> item
        ));
    }

    @Override
    public Map<String, DmpMeta> getMeta(String dmpLogin) {
        ListF<DmpMeta> result = Cf.arrayList();
        YPath metaTable = pathMeta.child(dmpLogin);

        if (yt.getHahn().cypress().exists(metaTable)) {
            yt.getHahn().tables().read(metaTable,
                    YTableEntryTypes.yson(DmpMeta.class), (Consumer<DmpMeta>) result::add);
        } else {
            LOG.warn("DMP meta for login " + dmpLogin + " is absent.");
        }

        return result.stream().collect(Collectors.toMap(
                item -> String.valueOf(item.getId()),
                item -> item
        ));
    }

    @Override
    public Map<String, DmpIndex> getIndexCached() {
        return Caching.fetch(indexCache, 0, this::getIndex);
    }

    @Override
    public LoadingCache<String, Map<String, DmpMeta>> getMetaCache() {
        return metaCache;
    }

    @Override
    public void refreshMetaCache() {
        getIndexCached()
                .values()
                .stream()
                .map(DmpIndex::getDmpLogin).collect(Collectors.toList())
                .forEach(metaCache::refresh);
    }
}
