package ru.yandex.crypta.service.entity.extractor;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.protobuf.util.JsonFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.ads.quality.adv.machine.entity.extract.protos.TCategoryDesc;
import ru.yandex.ads.quality.adv.machine.entity.extract.protos.TExtractorConfig;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;


public class YtEntityExtractorService implements EntityExtractorService {
    static private final String configPath = "entity_extract/config/config.json";
    static private final Integer dummyKey = 0;

    private final YtService yt;
    private final ExecutorService parentExecutor = Executors.newSingleThreadExecutor();
    private final ListeningExecutorService executorService = MoreExecutors.listeningDecorator(parentExecutor);

    private final LoadingCache<Integer, Map<String, Map<Long, String>>> namesCache = CacheBuilder.newBuilder()
            .refreshAfterWrite(1, TimeUnit.HOURS)
            .build(
                    new CacheLoader<Integer, Map<String, Map<Long, String>>>() {
                        @Override
                        public Map<String, Map<Long, String>> load(Integer key) {
                            return getNames();
                        }

                        @Override
                        public ListenableFuture<Map<String, Map<Long, String>>> reload(
                                final Integer key, Map<String, Map<Long, String>> oldValue
                        ) {
                            return executorService.submit(() -> load(key));
                        }
                    }
            );

    private static final Logger LOG = LoggerFactory.getLogger(YtEntityExtractorService.class);

    @Inject
    public YtEntityExtractorService(YtService yt) {
        this.yt = yt;
    }

    public static TExtractorConfig getExtractorConfig() throws IOException, URISyntaxException {
        var resource = YtEntityExtractorService.class.getClassLoader().getResourceAsStream(configPath);

        var configBuilder = TExtractorConfig.newBuilder();
        assert resource != null;
        JsonFormat.parser().ignoringUnknownFields().merge(new InputStreamReader(resource, StandardCharsets.UTF_8), configBuilder);
        return configBuilder.build();
    }

    private Map<String, Map<Long, String>> getNames() {
        try {
            var config = getExtractorConfig();

            return config.getCategoryDescsList().stream().parallel()
                    .map(this::readTable)
                    .flatMap(map -> map.entrySet().stream())
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        } catch (IOException | URISyntaxException e) {
            LOG.error("Failed to load local config: {}", e.getMessage());
            return new HashMap<>();
        }
    }

    private Map<String, Map<Long, String>> readTable(TCategoryDesc categoryDescs) {
        var result = new HashMap<String, Map<Long, String>>();

        yt.getHahn().tables().read(
                YPath.simple(categoryDescs.getSourceTable()),
                YTableEntryTypes.YSON,
                row -> {
                    var type = getRowType(categoryDescs, row);
                    var names = result.computeIfAbsent(type, key -> new HashMap<>());
                    names.put(row.getLong(categoryDescs.getIdField()), row.getString(categoryDescs.getTextField()));
                }
        );
        return result;
    }

    private String getRowType(TCategoryDesc categoryDescs, YTreeMapNode row) {
        if (categoryDescs.getTypeMapping().getMapCount() > 0) {
            var mapFieldValue = row.getString(categoryDescs.getTypeMapping().getMapField());
            return categoryDescs.getTypeMapping().getMapList().stream().filter(entry -> entry.getKey().equals(mapFieldValue)).reduce((a, b) -> {
                throw new IllegalStateException("Multiple elements: " + a + ", " + b);
            }).orElseThrow().getType().name();
        } else {
            return categoryDescs.getType().name();
        }
    }

    @Override
    public Map<String, Map<Long, String>> getNamesCached() {
        return Caching.fetchLoading(namesCache, dummyKey);
    }

    @Override
    public void refresh() {
        namesCache.refresh(dummyKey);
    }
}
