package ru.yandex.crypta.service.me;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableMap;

import ru.yandex.crypta.clients.bigb.BigbClient;
import ru.yandex.crypta.clients.bigb.BigbIdType;
import ru.yandex.crypta.clients.bigb.ProfileService;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.SegmentGroup;
import ru.yandex.crypta.lab.utils.SegmentName;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.proto.Me.Profile;
import ru.yandex.crypta.proto.Me.Profile.DmpSegment;
import ru.yandex.crypta.proto.Me.Profile.DmpSegment.Hierarchy;
import ru.yandex.crypta.proto.Me.Profile.DmpSegment.Status;
import ru.yandex.crypta.proto.Me.Profile.DmpSegment.Title;
import ru.yandex.crypta.proto.Me.Profile.DmpSegments;
import ru.yandex.crypta.proto.Me.Profile.ExactAge;
import ru.yandex.crypta.proto.Me.Profile.ExactGender;
import ru.yandex.crypta.proto.Me.Profile.ExactIncome;
import ru.yandex.crypta.proto.Me.Profile.HeuristicCommonInterest;
import ru.yandex.crypta.proto.Me.Profile.HeuristicOrganization;
import ru.yandex.crypta.proto.Me.Profile.InterestName;
import ru.yandex.crypta.proto.Me.Profile.LongtermInterests.Item;
import ru.yandex.crypta.proto.Me.Profile.Point;
import ru.yandex.crypta.proto.Me.Profile.ProfileSegment;
import ru.yandex.crypta.proto.Me.Profile.ShorttermInterest;
import ru.yandex.crypta.service.dmp.DmpIndex;
import ru.yandex.crypta.service.dmp.DmpMeta;
import ru.yandex.crypta.service.dmp.DmpService;
import ru.yandex.crypta.service.me.keyword.AllIdentifiersKeyword;
import ru.yandex.crypta.service.me.keyword.DmpKeyword;
import ru.yandex.crypta.service.me.keyword.GeoKeyword;
import ru.yandex.crypta.service.me.keyword.IndeviceIdentifiersKeyword;
import ru.yandex.crypta.service.me.keyword.Keyword;
import ru.yandex.crypta.service.me.keyword.SimpleKeyword;
import ru.yandex.crypta.service.me.keyword.WeightedKeyword;

public class ProfileMappingService implements ProfileService {
    private static final String EN_TAG = "EN";
    private static final String RU_TAG = "RU";
    private static final String GENDER_KEYWORD = "174";
    private static final String INCOME_5_KEYWORD = "614";
    private static final String AGE_6_KEYWORD = "543";
    private final static String ORGANIZATIONS_SEGMENT_ID_TEST = "group-20c7a93d";
    private final static String ORGANIZATIONS_SEGMENT_ID_PROD = "group-8130912a";

    private final BigbClient bigb;
    private final DmpService dmp;
    private final LabService lab;
    private final Cache<Integer, Map<Long, Map<Long, Segment>>> exportsCache =
            CacheBuilder.newBuilder()
                    .expireAfterWrite(5, TimeUnit.MINUTES)
                    .build();
    private String organizationsSegmentId;

    @Inject
    public ProfileMappingService(
            BigbClient bigb,
            DmpService dmp,
            LabService lab,
            EEnvironment environment
    )
    {
        this.bigb = bigb;
        this.dmp = dmp;
        this.lab = lab;

        if (Objects.equals(environment, EEnvironment.ENV_PRODUCTION)) {
            this.organizationsSegmentId = ORGANIZATIONS_SEGMENT_ID_PROD;
        } else {
            this.organizationsSegmentId = ORGANIZATIONS_SEGMENT_ID_TEST;
        }
    }

    private LabService lab(Language language) {
        return lab.withLanguage(Language.orDefault(language));
    }

    private Map<Long, Map<Long, Segment>> getExportsToSimpleSegments(Language language) {
        return Caching.fetch(exportsCache, 0, () -> lab(language).segments().getExportsToSimpleSegments());
    }

    private Map<String, Keyword> createMappingBuilder(Profile.Builder builder, Language language) {
        Map<Long, Map<Long, Segment>> exportsNames = getExportsToSimpleSegments(language);
        Map<String, List<SegmentGroup>> parents = lab(language).segments().getParentsPerSegment();

        Keyword genderKeyword = createGenderMapping(builder);
        Keyword ageKeyword = createAgeMapping(builder);
        Keyword incomeKeyword = createIncomeMapping(builder);
        Keyword loyaltyKeyword = createLoyaltyMapping(builder);
        Keyword probabilisticKeyword = createSegmentsMapping(builder, exportsNames);
        Keyword heuristicKeyword = createHeuristicMapping(builder, exportsNames);
        Keyword dmpKeyword = createDmpMapping(builder);
        Keyword longtermInterestsKeyword = createLongtermInterestsMapping(builder, exportsNames);
        Keyword shorttermInterestsKeyword = createShorttermInterestsMapping(builder, exportsNames);
        Keyword exactDemographicsKeyword = createExactDemographicsMapping(builder);
        Keyword newHeuristicKeyword = createNewHeuristicMapping(builder, exportsNames, parents);
        Keyword indeviceIdentifiersKeyword = createIndeviceIdentifiersMapping(builder);
        Keyword allIdentifiersKeyword = createAllIdentifiersMapping(builder);
        Keyword regularGeoKeyword = createRegularGeoMapping(builder);
        Keyword actualGeoKeyword = createActualGeoMapping(builder);

        return ImmutableMap.<String, Keyword>builder()
                .put(GENDER_KEYWORD, genderKeyword)
                .put("216", heuristicKeyword)
                .put("217", probabilisticKeyword)
                .put("220", loyaltyKeyword)
                .put(AGE_6_KEYWORD, ageKeyword)
                .put(INCOME_5_KEYWORD, incomeKeyword)
                .put("485", dmpKeyword)
                .put("601", longtermInterestsKeyword)
                .put("602", shorttermInterestsKeyword)
                .put("569", exactDemographicsKeyword)
                .put("547", newHeuristicKeyword)
                .put(IndeviceIdentifiersKeyword.ID, indeviceIdentifiersKeyword)
                .put(GeoKeyword.REGULAR_GEO_ID, regularGeoKeyword)
                .put(GeoKeyword.ACTUAL_GEO_ID, actualGeoKeyword)
                .put(AllIdentifiersKeyword.ID, allIdentifiersKeyword)
                .build();
    }

    private SimpleKeyword createNewHeuristicMapping(
            Profile.Builder builder, Map<Long, Map<Long, Segment>> exportsNames,
            Map<String, List<SegmentGroup>> parents
    )
    {
        Profile.NewHeuristicInterests.Builder heuristicCommmon = builder.getNewHeuristicBuilder();
        Map<Long, Segment> newHeuristicSegments = exportsNames.getOrDefault(547L, Collections.emptyMap());

        return SimpleKeyword.with(each -> {
            long timestamp = Keyword.getTimestamp(each);
            List<Integer> ids = Arrays.stream(Keyword.getValue(each).split(","))
                    .map(Integer::valueOf)
                    .collect(Collectors.toList());

            ids.forEach(id -> {
                Profile.HeuristicCommonInterest.Builder common = HeuristicCommonInterest.newBuilder();
                Profile.HeuristicOrganization.Builder organization = HeuristicOrganization.newBuilder();
                Segment segment = newHeuristicSegments.get(Long.valueOf(id));

                if (Objects.nonNull(segment)) {
                    List<SegmentGroup> segmentParents = parents.get(segment.getId());

                    if (segmentParents.size() > 2 && segmentParents.get(1).getId().equals(organizationsSegmentId)) {
                        organization.setId(id);
                        organization.setName(segment.getName());
                        organization.setDescription(segment.getDescription());
                        organization.setSegmentId(segment.getId());
                        heuristicCommmon.addOrganizations(organization);
                    } else {
                        common.setId(id);
                        common.setName(segment.getName());
                        common.setDescription(segment.getDescription());
                        common.setSegmentId(segment.getId());
                        heuristicCommmon.addCommon(common);
                    }
                } else {
                    common.setId(id);
                    common.setName("Name unknown for id=" + id);
                    common.setDescription("Description unknown for id=" + id);
                    common.setSegmentId("Unknown");
                    heuristicCommmon.addCommon(common);
                }

                heuristicCommmon.setTimestamp(timestamp);
            });
        });
    }

    private SimpleKeyword createHeuristicMapping(
            Profile.Builder builder,
            Map<Long, Map<Long, Segment>> exportsNames
    )
    {
        Long keywordId = 216L;
        Map<Long, Segment> segmentNames = exportsNames.getOrDefault(keywordId, Collections.emptyMap());
        Profile.ProfileSegments.Builder segments = builder.getHeuristicBuilder();

        return SimpleKeyword.with(
                each -> {
                    long timestamp = Keyword.getTimestamp(each);
                    List<Long> ids = Arrays.stream(Keyword.getValue(each).split(","))
                            .map(value -> Long.valueOf(value.split(":", -1)[0]))
                            .collect(Collectors.toList());

                    ids.forEach(id -> {
                        Segment simpleSegment = segmentNames.get(id);

                        if (Objects.nonNull(simpleSegment)) {
                            Profile.ProfileSegment.Builder segment = ProfileSegment.newBuilder();
                            segment.setName(simpleSegment.getName())
                                    .setDescription(simpleSegment.getDescription())
                                    .setId(id)
                                    .setSegmentId(simpleSegment.getId());
                            segments.addValue(segment);
                        }
                    });

                    segments.setTimestamp(timestamp);
                }
        );
    }

    private SimpleKeyword createSegmentsMapping(
            Profile.Builder builder,
            Map<Long, Map<Long, Segment>> exportsNames
    )
    {
        return SimpleKeyword.with(
                (each) -> {
                    Long id = Long.valueOf(Keyword.beforeSemicolon(Keyword.getValue(each)));
                    long timestamp = Keyword.getTimestamp(each);
                    Segment simpleSegment = exportsNames
                            .getOrDefault(217L, Collections.emptyMap())
                            .get(id);
                    if (Objects.nonNull(simpleSegment)) {
                        Profile.ProfileSegments.Builder segments = null;
                        switch (simpleSegment.getType()) {
                            case INTEREST:
                                segments = builder.getInterestBuilder();
                                break;
                            case LOOKALIKE:
                                segments = builder.getProbabilisticBuilder();
                                break;
                            default:
                                break;
                        }
                        if (Objects.nonNull(segments)) {
                            segments.setTimestamp(timestamp)
                                    .addValueBuilder()
                                    .setName(simpleSegment.getName())
                                    .setDescription(simpleSegment.getDescription())
                                    .setId(id)
                                    .setSegmentId(simpleSegment.getId())
                                    .setWeight(Keyword.getWeight(each));
                        }
                    }
                }
        );
    }

    private Keyword createExactDemographicsMapping(Profile.Builder builder) {
        Profile.ExactDemographics.Builder demographics = builder.getExactDemographicsBuilder();
        return SimpleKeyword.with(each -> {
            Long timestamp = SimpleKeyword.getTimestamp(each);
            List<String> values = Arrays.asList(SimpleKeyword.getValue(each).split(","));
            Map<String, Integer> valueMap = new HashMap<>();

            values.forEach(value -> {
                List<String> pair = Arrays.asList(value.split(":"));
                valueMap.put(pair.get(0), Integer.valueOf(pair.get(1)));
            });

            demographics
                    .setTimestamp(timestamp)
                    .setGenderValue(Optional.ofNullable(valueMap.get(GENDER_KEYWORD)).map(value -> value + 1)
                            .orElse(ExactGender.unknownGender_VALUE))
                    .setIncome5Value(Optional.ofNullable(valueMap.get(INCOME_5_KEYWORD)).map(value -> value + 1)
                            .orElse(ExactIncome.unknownIncome_VALUE))
                    .setAge6Value(Optional.ofNullable(valueMap.get(AGE_6_KEYWORD)).map(value -> value + 1)
                            .orElse(ExactAge.unknownAge_VALUE));

        });

    }

    private DmpKeyword createDmpMapping(Profile.Builder builder) {
        Map<String, DmpIndex> dmpIndex = dmp.getIndexCached();
        Profile.Dmp.Builder segmentBin = builder.getDmpBuilder();

        return DmpKeyword.with(
                (each) -> {
                    Integer id = DmpKeyword.getDmpId(each);
                    Long timestamp = DmpKeyword.getTimestamp(each);
                    List<String> values = Arrays.asList(DmpKeyword.getValue(each).split(","));

                    String dmpLogin = dmpIndex.get(id.toString()).getDmpLogin();
                    String dmpName = dmpIndex.get(id.toString()).getName();

                    Profile.DmpSegments.Builder dmpSegments = DmpSegments.newBuilder();
                    dmpSegments
                            .setId(id)
                            .setLogin(dmpLogin)
                            .setName(dmpName)
                            .setTimestamp(timestamp);

                    Map<String, DmpMeta> dmpMeta = Caching.fetchLoading(dmp.getMetaCache(), dmpLogin);

                    List<DmpSegment> segments = values.stream().map(value -> {
                        DmpMeta meta = dmpMeta.get(value);

                        if (Objects.isNull(meta)) {
                            return DmpSegment.newBuilder().setId(Integer.valueOf(value)).build();
                        }

                        Status status;
                        switch (meta.getStatus()) {
                            case "disabled":
                                status = Status.DISABLED;
                                break;
                            case "enabled":
                                status = Status.ENABLED;
                                break;
                            case "deleted":
                                status = Status.DELETED;
                                break;
                            default:
                                status = Status.DISABLED;
                                break;
                        }

                        Title title = extractTitle(meta.getTitle());
                        Hierarchy hierarchy = extractHierarchy(meta.getHierarchy());
                        DmpSegment.Builder segment = DmpSegment.newBuilder();

                        return segment
                                .setId(Integer.valueOf(value))
                                .setTimestamp(meta.getTimestamp())
                                .setStatus(status)
                                .setTitle(title)
                                .setHierarchy(hierarchy)
                                .build();
                    }).collect(Collectors.toList());

                    dmpSegments.addAllSegments(segments);

                    segmentBin.addValue(dmpSegments);
                }
        );
    }

    private Title extractTitle(Map<String, String> dmpTitle) {
        Profile.DmpSegment.Title.Builder builder = DmpSegment.Title.newBuilder();
        List<String> keys = new ArrayList<>(dmpTitle.keySet());

        if (keys.size() == 1) {
            return builder.setFallback(dmpTitle.get(keys.get(0))).build();
        }

        keys.forEach(key -> {
            if (key.toUpperCase().contains(EN_TAG) && dmpTitle.get(key) != null) {
                builder.setEn(dmpTitle.get(key));
            }

            if (key.toUpperCase().contains(RU_TAG) && dmpTitle.get(key) != null) {
                builder.setRu(dmpTitle.get(key));
            }
        });

        return builder.build();
    }

    private Hierarchy extractHierarchy(Map<String, List<String>> dmpHierarchy) {
        Profile.DmpSegment.Hierarchy.Builder builder = DmpSegment.Hierarchy.newBuilder();
        List<String> keys = new ArrayList<>(dmpHierarchy.keySet());

        if (keys.size() == 1) {
            return builder.addAllFallback(dmpHierarchy.get(keys.get(0))).build();
        }

        keys.forEach(key -> {
            if (key.toUpperCase().contains(EN_TAG) && dmpHierarchy.get(key) != null) {
                builder.addAllEn(dmpHierarchy.get(key));
            }

            if (key.toUpperCase().contains(RU_TAG) && dmpHierarchy.get(key) != null) {
                builder.addAllRu(dmpHierarchy.get(key));
            }
        });

        return builder.build();
    }

    private IndeviceIdentifiersKeyword createIndeviceIdentifiersMapping(Profile.Builder builder) {
        Profile.IndeviceIdentifiers.Builder identifiers = builder.getIndeviceIdentifiersBuilder();
        return IndeviceIdentifiersKeyword.with(each -> identifiers
                .setCryptaId(IndeviceIdentifiersKeyword.getCryptaId(each))
        );
    }

    private AllIdentifiersKeyword createAllIdentifiersMapping(Profile.Builder builder) {
        Profile.AllIdentifiers.Builder identifiers = builder.getAllIdentifiersBuilder();
        return new AllIdentifiersKeyword(each ->
                AllIdentifiersKeyword.getIds(each)
                        .forEachRemaining(
                                idRecord -> {
                                    var addedId = identifiers.addIdentifiersBuilder();
                                    addedId.setId(AllIdentifiersKeyword.getIdValue(idRecord));
                                    addedId.setType(AllIdentifiersKeyword.getIdType(idRecord));
                                }
                        ));
    }

    private GeoKeyword createRegularGeoMapping(Profile.Builder builder) {
        return GeoKeyword.with(each -> {
            GeoKeyword.getRegular(each).forEach(pointNode -> {
                Profile.Point.Builder point = Point.newBuilder();
                point.setLatitude(Float.valueOf(pointNode.get("latitude").asText()));
                point.setLongitude(Float.valueOf(pointNode.get("longitude").asText()));
                point.setTimestamp(pointNode.get("update_time").asLong());
                builder.addRegularGeo(point.build());
            });
        });
    }

    private float convertGeoToNormal(long coordinate) {
        return coordinate / 10000000f - 180;
    }

    private GeoKeyword createActualGeoMapping(Profile.Builder builder) {
        return GeoKeyword.with(each -> {
            Profile.Point.Builder point = Point.newBuilder();
            List<Long> coordinates = GeoKeyword.getActual(each);
            point.setLongitude(convertGeoToNormal(coordinates.get(0)));
            point.setLatitude(convertGeoToNormal(coordinates.get(1)));
            point.setTimestamp(GeoKeyword.getTimestamp(each));
            builder.addActualGeo(point);
        });
    }

    private SimpleKeyword createLoyaltyMapping(Profile.Builder builder) {
        Profile.Loyalty.Builder loyalty = builder.getLoyaltyBuilder();
        return SimpleKeyword.with(
                (each) -> loyalty
                        .setValue(Keyword.normalize(Float.valueOf(Keyword.getValue(each))))
                        .setTimestamp(Keyword.getTimestamp(each))
        );
    }

    private WeightedKeyword createIncomeMapping(Profile.Builder builder) {
        Profile.Income.Builder income = builder.getIncomeBuilder();
        return WeightedKeyword.builder()
                .timestamp(income::setTimestamp)
                .value("0", income::setA)
                .value("1", income::setB1)
                .value("2", income::setB2)
                .value("3", income::setC1)
                .value("4", income::setC2)
                .build();
    }

    private WeightedKeyword createGenderMapping(Profile.Builder builder) {
        Profile.Gender.Builder gender = builder.getGenderBuilder();
        return WeightedKeyword.builder()
                .timestamp(gender::setTimestamp)
                .value("0", gender::setMale)
                .value("1", gender::setFemale)
                .build();
    }

    private WeightedKeyword createAgeMapping(Profile.Builder builder) {
        Profile.Age.Builder age = builder.getAgeBuilder();
        return WeightedKeyword.builder()
                .timestamp(age::setTimestamp)
                .value("0", age::setFrom0To17)
                .value("1", age::setFrom18To24)
                .value("2", age::setFrom25To34)
                .value("3", age::setFrom35To44)
                .value("4", age::setFrom45To54)
                .value("5", age::setFrom55To99)
                .build();
    }

    private String extractSegmentId(Map<Long, Segment> segments, Long id) {
        Segment segment = segments.get(id);

        if (segment == null) {
            return "";
        }

        return segment.getId();
    }

    private SimpleKeyword createLongtermInterestsMapping(Profile.Builder builder,
            Map<Long, Map<Long, Segment>> exportsNames)
    {
        Profile.LongtermInterests.Builder interests = builder.getLongtermBuilder();
        Map<Long, SegmentName> names = lab.segments().getSegmentNamesWithExportId(601L);
        Map<Long, Segment> longtermSegments = exportsNames.getOrDefault(601L, Collections.emptyMap());

        return SimpleKeyword.with(each -> {
            long timestamp = SimpleKeyword.getTimestamp(each);
            List<String> values = Arrays.asList(SimpleKeyword.getValue(each).split(","));

            List<Item> items = values.stream().map(value -> {
                Item.Builder interest = Item.newBuilder();

                return interest
                        .setId(Integer.valueOf(value))
                        .setName(extractInterestName(names, Long.valueOf(value)))
                        .setSegmentId(extractSegmentId(longtermSegments, Long.valueOf(value)))
                        .build();
            }).collect(Collectors.toList());

            interests.setTimestamp(timestamp).addAllValue(items);
        });
    }

    private SimpleKeyword createShorttermInterestsMapping(Profile.Builder builder,
            Map<Long, Map<Long, Segment>> exportsNames)
    {
        Map<Long, SegmentName> names = lab.segments().getSegmentNamesWithExportId(602L);
        Map<Long, Segment> shorttermSegments = exportsNames.getOrDefault(602L, Collections.emptyMap());
        Profile.ShorttermInterests.Builder interests = builder.getShorttermBuilder();

        return SimpleKeyword.with(each -> {
            long timestamp = SimpleKeyword.getTimestamp(each);
            int id = Integer.parseInt(SimpleKeyword.getValue(each));

            interests.addValue(ShorttermInterest.newBuilder()
                    .setId(id)
                    .setName(extractInterestName(names, Long.valueOf(id)))
                    .setSegmentId(extractSegmentId(shorttermSegments, Long.valueOf(id)))
                    .setTimestamp(timestamp)
                    .build());
        });
    }

    private InterestName extractInterestName(Map<Long, SegmentName> segmentNames, Long segmentId) {
        SegmentName segmentName = segmentNames.get(segmentId);
        InterestName.Builder interestName = InterestName.newBuilder();
        String nameEn = "";
        String nameRu = "";

        if (segmentName != null) {
            Map<String, String> name = segmentName.getName();
            nameEn = name.get("en");
            nameRu = name.get("ru");
        }

        return interestName.setEn(nameEn).setRu(nameRu).build();
    }

    private Consumer<JsonNode> acceptInto(Profile.Builder builder, Language language) {
        Map<String, Keyword> keywords = createMappingBuilder(builder, language);
        return jsonNode -> {
            String id = jsonNode.get("id").asText();
            Keyword keyword = keywords.get(id);
            if (keyword != null) {
                keyword.accept(jsonNode);
            }
        };
    }

    private Iterator<JsonNode> getSegments(BigbIdType bigbIdType, String bigIdValue) {
        return bigb.getBigbData(bigbIdType, bigIdValue).get("data").get(0).get("segment").elements();
    }

    @Override
    public Profile get(BigbIdType bigbIdType, String bigbIdValue, Language language) {
        var profile = Profile.newBuilder();
        getSegments(bigbIdType, bigbIdValue).forEachRemaining(acceptInto(profile, language));
        return profile.build();
    }

}
