package ru.yandex.crypta.service.pages;


import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import org.jetbrains.annotations.NotNull;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.proto.TYqlConfig;
import ru.yandex.yql.YqlDataSource;
import ru.yandex.yql.settings.YqlProperties;

public class DefaultPageService implements PageService {

    private static final String URLS_PAGE_IDS = "//home/yabs/dict/Page";
    private static final String FIELD_SITE_NAME = "Name";
    private static final String FIELD_PAGE_ID = "PageID";
    private static final String FIELD_DESCRIPTION = "Description";

    private final TYqlConfig yqlConfig;

    private final LoadingCache<Integer, PageInfo> allPagesCache = CacheBuilder.newBuilder()
            .refreshAfterWrite(1, TimeUnit.DAYS)
            .build(new CacheLoader<>() {
                @Override
                public PageInfo load(@NotNull Integer key) {
                    return getAll();
                }

                @Override
                public ListenableFuture<PageInfo> reload(
                        @NotNull final Integer key, @NotNull PageInfo oldValue
                ) {
                    // Load new values asynchronously and non-blocking
                    return executorService.submit(() -> load(0));
                }
            });
    private final ExecutorService parentExecutor = Executors.newSingleThreadExecutor();
    private final ListeningExecutorService executorService = MoreExecutors.listeningDecorator(parentExecutor);

    @Inject
    public DefaultPageService(TYqlConfig yqlConfig) {
        this.yqlConfig = yqlConfig;
    }

    @Override
    public PageInfo getAll() {
        YqlDataSource yqlDataSource = new YqlDataSource(
                yqlConfig.getConnectionString(),
                new YqlProperties().withCredentials(yqlConfig.getUser(), yqlConfig.getToken())
        );

        String query = String.format(
                "SELECT %2$s, %3$s, %4$s FROM `%1$s`;",
                URLS_PAGE_IDS,
                FIELD_SITE_NAME,
                FIELD_PAGE_ID,
                FIELD_DESCRIPTION
        );

        Map<Long, String> pages = new HashMap<>();
        Map<Long, String> pageDescriptions = new HashMap<>();

        Connection connection = null;
        Statement statement = null;

        try {
            connection = yqlDataSource.getConnection();
            statement = connection.createStatement();

            ResultSet rs = statement.executeQuery(query);

            while (rs.next()) {
                pages.put(
                        rs.getLong(FIELD_PAGE_ID),
                        rs.getString(FIELD_SITE_NAME)
                );
                pageDescriptions.put(
                        rs.getLong(FIELD_PAGE_ID),
                        rs.getString(FIELD_DESCRIPTION)
                );
            }

            rs.close();
            statement.close();
            connection.close();
        } catch (SQLException sqle) {
            throw Exceptions.internal("YQL query failed:\n" + sqle.getMessage());
        } catch (Exception e){
            if (statement != null) {
                try {
                    statement.close();
                } catch (SQLException closeStatementException) {
                    throw Exceptions.internal("Failed to fulfill statement:\n" + closeStatementException);
                }
            }

            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException closeConnectionException) {
                    throw Exceptions.internal("Failed to close connection:\n" + closeConnectionException);
                }
            }

        }

        var siteIds = pages.entrySet().stream().collect(Collectors.groupingBy(
                Map.Entry::getValue,
                Collectors.mapping(Map.Entry::getKey, Collectors.toList())
        ));
        return new PageInfo(siteIds, pages, pageDescriptions);
    }

    @Override
    public LoadingCache<Integer, PageInfo> getAllPagesCache() {
        return allPagesCache;
    }

    @Override
    public void refreshAllPagesCache() {
        allPagesCache.refresh(0);
    }
}
