package ru.yandex.crypta.service.public_profile;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableMap;

import ru.yandex.crypta.clients.bigb.BigbClient;
import ru.yandex.crypta.clients.bigb.BigbIdType;
import ru.yandex.crypta.clients.bigb.PublicProfileService;
import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.SegmentGroup;
import ru.yandex.crypta.lab.utils.SegmentName;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.proto.PublicProfile;
import ru.yandex.crypta.service.me.keyword.GeoKeyword;
import ru.yandex.crypta.service.me.keyword.Keyword;
import ru.yandex.crypta.service.me.keyword.SimpleKeyword;


public class PublicProfileMappingService implements PublicProfileService {
    private static final String GENDER_KEYWORD = "174";
    private static final String INCOME_5_KEYWORD = "614";
    private static final String AGE_6_KEYWORD = "543";
    private static final Long HEURICTIC_KEYWORD = 547L;
    private static final Long LONGTERM_KEYWORD = 601L;
    private static final Long SHORTTERM_KEYWORD = 602L;
    private final static String ORGANIZATIONS_GROUP_ID_TEST = "group-20c7a93d";
    private final static String ORGANIZATIONS_GROUP_ID_PROD = "group-8130912a";
    private final static String ZODIAC_SIGNS_GROUP_ID_TEST = "group-20c7a93d";
    private final static String ZODIAC_SIGNS_GROUP_ID_PROD = "group-17d11adb";
    private final static String INTERESTS_GROUP_ID_TEST = "group-ff85882a";
    private final static String INTERESTS_GROUP_ID_PROD = "group-d0802a75";
    private final static Integer HAS_CHILDREN_ID = 1919;
    private final static Integer MARRIED_ID = 1023;
    private final static Integer NOT_MARRIED_ID = 1024;

    private final BigbClient bigb;
    private final LabService lab;
    private final Cache<Integer, Map<Long, Map<Long, Segment>>> exportsCache =
            CacheBuilder.newBuilder()
                    .expireAfterWrite(5, TimeUnit.MINUTES)
                    .build();
    private final Cache<Integer, Map<String, List<SegmentGroup>>> parentsCache =
            CacheBuilder.newBuilder()
                    .expireAfterWrite(5, TimeUnit.MINUTES)
                    .build();
    private final String organizationsGroupId;
    private final String zodiacSignsGroupId;
    private final String interestsGroupId;
    private final Integer hasChildrenId;
    private final Integer marriedId;
    private final Integer notMarriedId;

    @Inject
    public PublicProfileMappingService(
            BigbClient bigb,
            LabService lab,
            EEnvironment environment
    )
    {
        this.bigb = bigb;
        this.lab = lab;

        if (Objects.equals(environment, EEnvironment.ENV_PRODUCTION)) {
            this.organizationsGroupId = ORGANIZATIONS_GROUP_ID_PROD;
            this.zodiacSignsGroupId = ZODIAC_SIGNS_GROUP_ID_PROD;
            this.interestsGroupId = INTERESTS_GROUP_ID_PROD;
            this.hasChildrenId = HAS_CHILDREN_ID;
            this.marriedId = MARRIED_ID;
            this.notMarriedId = NOT_MARRIED_ID;
        } else {
            this.organizationsGroupId = ORGANIZATIONS_GROUP_ID_TEST;
            this.zodiacSignsGroupId = ZODIAC_SIGNS_GROUP_ID_TEST;
            this.interestsGroupId = INTERESTS_GROUP_ID_TEST;
            this.hasChildrenId = HAS_CHILDREN_ID;
            this.marriedId = MARRIED_ID;
            this.notMarriedId = NOT_MARRIED_ID;
        }
    }

    private LabService lab(Language language) {
        return lab.withLanguage(Language.orDefault(language));
    }

    private Map<Long, Map<Long, Segment>> getExportsToSimpleSegments(Language language) {
        return Caching.fetch(exportsCache, 0, () -> lab(language).segments().getExportsToSimpleSegments());
    }

    private Map<String, List<SegmentGroup>> getSegmentsParents(Language language) {
        return Caching.fetch(parentsCache, 0, () -> lab(language).segments().getParentsPerSegment());
    }

    private Map<String, Integer> countInterestsPerGroup(Map<String, List<SegmentGroup>> parents, Map<Long, Map<Long, Segment>> exportsNames) {
        Map<String, Integer> interestsPerGroupCounts  = new HashMap<>();
        var interestsSegments = exportsNames.getOrDefault(LONGTERM_KEYWORD, Collections.emptyMap());
        interestsSegments.forEach((key, value) -> {
            var interestSegmentId = value.getId();
            var interestParents = parents.get(interestSegmentId);

            if (interestParents.size() < 3 || !interestParents.get(interestParents.size()-3).getId().equals(interestsGroupId)) {
                throw Exceptions.notFound(String.format("Segment with id=%s is not from interests group (id=%s) ", value.getId(), interestsGroupId));
            }

            String groupId;
            if (interestParents.size() == 3) {
                groupId = interestSegmentId;
            } else {
                groupId = interestParents.get(interestParents.size()-4).getId();
            }

            var groupName = lab(Language.RU).segments().get(groupId).getName();
            interestsPerGroupCounts.put(groupName, interestsPerGroupCounts.getOrDefault(groupName, 0) + 1);
        });
        return interestsPerGroupCounts;
    }

    private Map<String, Keyword> createMappingBuilder(
            PublicProfile.Builder builder,
            Map<Long, Map<Long, Segment>> exportsNames,
            Map<String, List<SegmentGroup>> parents
    ) {
        HashSet<Integer> includedInterests = new HashSet<>();

        Keyword longtermInterestsKeyword = createInterestsMapping(builder, LONGTERM_KEYWORD, exportsNames, parents, includedInterests);
        Keyword shorttermInterestsKeyword = createInterestsMapping(builder, SHORTTERM_KEYWORD, exportsNames, parents, includedInterests);
        Keyword exactDemographicsKeyword = createExactDemographicsMapping(builder);
        Keyword heuristicKeyword = createHeuristicMapping(builder, exportsNames, parents);
        Keyword regularGeoKeyword = createRegularGeoMapping(builder);

        return ImmutableMap.<String, Keyword>builder()
                .put("601", longtermInterestsKeyword)
                .put("602", shorttermInterestsKeyword)
                .put("569", exactDemographicsKeyword)
                .put("547", heuristicKeyword)
                .put(GeoKeyword.REGULAR_GEO_ID, regularGeoKeyword)
                .build();
    }

    private void getTopInterests(PublicProfile.Builder builder, Map<String, Integer> groupCounts) {
        Map<String, Float> userInterestsGroupsCounts = new HashMap<>();

        builder.getInterestsList().forEach(interest -> {
            var groupName = interest.getGroupName();
            userInterestsGroupsCounts.put(groupName, userInterestsGroupsCounts.getOrDefault(groupName, 0.0f) + 1);
            }
        );

        userInterestsGroupsCounts.forEach((key, value) -> userInterestsGroupsCounts.put(key, userInterestsGroupsCounts.get(key) / (float) groupCounts.get(key)));

        userInterestsGroupsCounts.entrySet().stream()
            .sorted(Map.Entry.comparingByValue(Comparator.reverseOrder()))
            .limit(3)
            .forEach(entry -> builder.addInterestsTop(entry.getKey()));
    }

    private SimpleKeyword createHeuristicMapping(
            PublicProfile.Builder builder, Map<Long, Map<Long, Segment>> exportsNames,
            Map<String, List<SegmentGroup>> parents
    )
    {
        Map<Long, Segment> heuristicSegments = exportsNames.getOrDefault(HEURICTIC_KEYWORD, Collections.emptyMap());

        return SimpleKeyword.with(each -> {
            List<Integer> ids = Arrays.stream(Keyword.getValue(each).split(","))
                    .map(Integer::valueOf)
                    .collect(Collectors.toList());

            ids.forEach(id -> {
                Segment segment = heuristicSegments.get(Long.valueOf(id));

                PublicProfile.Segment.Builder segmentBuilder = PublicProfile.Segment.newBuilder();

                if (Objects.nonNull(segment)) {
                    segmentBuilder
                            .setId(id)
                            .setName(segment.getName())
                            .setKeywordId(HEURICTIC_KEYWORD);

                    List<SegmentGroup> segmentParents = parents.get(segment.getId());
                    if (segmentParents.size() > 2 && segmentParents.get(1).getId().equals(organizationsGroupId)) {
                        builder.addOrganizations(segmentBuilder);
                    } else if (segmentParents.size() > 1 && segmentParents.get(0).getId().equals(zodiacSignsGroupId)) {
                        builder.setZodiacSign(segmentBuilder);
                    } else if (id.equals(hasChildrenId)) {
                        builder.setHasChildren(true);
                    } else if (id.equals(marriedId)) {
                        builder.setMarried(true);
                    } else if (id.equals(notMarriedId)) {
                        builder.setMarried(false);
                    } else {
                        builder.addHeuristic(segmentBuilder);
                    }
                } else {
                    segmentBuilder
                        .setId(id)
                        .setName("Name unknown for id=" + id)
                        .setKeywordId(HEURICTIC_KEYWORD);
                    builder.addHeuristic(segmentBuilder);
                }
            });
        });
    }

    private String extractInterestGroupName(
            Map<Long, Segment> interestsSegments,
            Map<String, List<SegmentGroup>> parents,
            Long interestId)
    {
        List<SegmentGroup> interestParents;
        Segment interest;
        try {
            interest = interestsSegments.get(interestId);
            interestParents = parents.get(interest.getId());
        } catch (NullPointerException e) {
            return "Unknown group name";
        }


        if (interestParents.size() < 3 || !interestParents.get(interestParents.size()-3).getId().equals(interestsGroupId)) {
            throw Exceptions.notFound(String.format("Segment with id=%d is not from interests group (id=%s) ", interestId, interestsGroupId));
        }

        String groupId;
        if (interestParents.size() == 3) {
            groupId = interest.getId();
        } else {
            groupId = interestParents.get(interestParents.size()-4).getId();
        }

        try {
            return lab(Language.RU).segments().get(groupId).getName();
        } catch (NotFoundException e) {
            return "Unknown group name";
        }
    }


    private String extractInterestName(Map<Long, SegmentName> interestsNames, Long interestId) {
        SegmentName interestName = interestsNames.get(interestId);
        String name = "Unknown name";

        if (interestName != null) {
            name = interestName.getName().get("ru");
        }
        return name;
    }

    private SimpleKeyword createInterestsMapping(
        PublicProfile.Builder builder, Long keywordId,
        Map<Long, Map<Long, Segment>> exportsNames,
        Map<String, List<SegmentGroup>> parents,
        HashSet<Integer> includedInterests
    )
    {
        var interestsSegments = exportsNames.getOrDefault(keywordId, Collections.emptyMap());
        Map<Long, SegmentName> interestsNames = lab.segments().getSegmentNamesWithExportId(keywordId);

        return SimpleKeyword.with(each -> {
            List<String> values = Arrays.asList(Keyword.getValue(each).split(","));

            values.forEach(value -> {
                Integer id = Integer.parseInt(value);

                if (includedInterests.contains(id)) {
                    return;
                }

                PublicProfile.Interest.Builder interest = PublicProfile.Interest.newBuilder();
                String groupName = extractInterestGroupName(interestsSegments, parents, Long.valueOf(value));

                interest
                    .setName(extractInterestName(interestsNames, Long.valueOf(value)))
                    .setGroupName(groupName);

                builder.addInterests(interest.build());
                includedInterests.add(id);
            });
        });
    }

    private SimpleKeyword createExactDemographicsMapping(PublicProfile.Builder builder) {
        PublicProfile.ExactDemographics.Builder demographics = builder.getExactDemographicsBuilder();
        return SimpleKeyword.with(each -> {
            List<String> values = Arrays.asList(Keyword.getValue(each).split(","));
            Map<String, Integer> valueMap = new HashMap<>();

            values.forEach(value -> {
                List<String> pair = Arrays.asList(value.split(":"));
                valueMap.put(pair.get(0), Integer.valueOf(pair.get(1)));
            });

            demographics
                    .setGenderValue(Optional.ofNullable(valueMap.get(GENDER_KEYWORD)).map(value -> value + 1)
                            .orElse(PublicProfile.ExactGender.unknownGender_VALUE))
                    .setIncomeValue(Optional.ofNullable(valueMap.get(INCOME_5_KEYWORD)).map(value -> value + 1)
                            .orElse(PublicProfile.ExactIncome.unknownIncome_VALUE))
                    .setAgeValue(Optional.ofNullable(valueMap.get(AGE_6_KEYWORD)).map(value -> value + 1)
                            .orElse(PublicProfile.ExactAge.unknownAge_VALUE));
        });
    }

    private GeoKeyword createRegularGeoMapping(PublicProfile.Builder builder) {
        return GeoKeyword.with(each -> {
            GeoKeyword.getRegular(each).forEach(pointNode -> {
                PublicProfile.Point.Builder point = PublicProfile.Point.newBuilder();
                point.setLatitude(Float.parseFloat(pointNode.get("latitude").asText()));
                point.setLongitude(Float.parseFloat(pointNode.get("longitude").asText()));
                point.setTimestamp(pointNode.get("update_time").asLong());
                point.setTypeValue(pointNode.get("type").asInt());
                builder.addRegularGeo(point.build());
            });
        });
    }

    private Consumer<JsonNode> acceptInto(
            PublicProfile.Builder builder,
            Map<Long, Map<Long, Segment>> exportsNames,
            Map<String, List<SegmentGroup>> parents
    ) {
        Map<String, Keyword> keywords = createMappingBuilder(builder, exportsNames, parents);
        return jsonNode -> {
            String id = jsonNode.get("id").asText();
            Keyword keyword = keywords.get(id);
            if (keyword != null) {
                keyword.accept(jsonNode);
            }
        };
    }

    private Iterator<JsonNode> getSegments(BigbIdType bigbIdType, String bigIdValue) {
        return bigb.getBigbData(bigbIdType, bigIdValue).get("data").get(0).get("segment").elements();
    }

    @Override
    public PublicProfile get(BigbIdType bigbIdType, String bigbIdValue, Language language) {
        var profile = PublicProfile.newBuilder();
        Map<Long, Map<Long, Segment>> exportsNames = getExportsToSimpleSegments(language);
        Map<String, List<SegmentGroup>> parents = getSegmentsParents(language);

        getSegments(bigbIdType, bigbIdValue).forEachRemaining(acceptInto(profile, exportsNames, parents));
        getTopInterests(profile, countInterestsPerGroup(parents, exportsNames));

        return profile.build();
    }

}
